# Evernote macOS 侧边栏架构文档

## 总体架构概览

Evernote macOS的侧边栏基于模块化设计，支持两种显示模式：
- **Outline View Mode**: 传统的树形结构导航视图
- **Stack View Mode**: 紧凑的垂直堆叠视图模式

## 核心组件架构

### 1. ENNavigationManager (导航管理器)
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Manager/`

**职责**：
- 管理所有导航项的可见性和隐藏状态
- 处理用户自定义隐藏的项目和系统默认隐藏的项目
- 协调两种导航模式之间的切换

**核心功能**：
```objc
// 预定义的导航项标识符
kENNavigationOutlineViewControllerItemIdentifierShortcutsRoot  // 快捷方式
kENNavigationOutlineViewControllerItemIdentifierAIRoot        // AI功能根节点
kENNavigationOutlineViewControllerItemIdentifierNarrowAIWrite // AI写作
kENNavigationOutlineViewControllerItemIdentifierNarrowAIRead  // AI阅读
kENNavigationOutlineViewControllerItemIdentifierNotes        // 笔记
kENNavigationOutlineViewControllerItemIdentifierTrash        // 垃圾箱
kENNavigationOutlineViewControllerItemIdentifierUpgrade      // 升级
```

**可见性管理**：
- `userHiddenItemIdentifiers`: 用户手动隐藏的项目
- `systemHiddenItemIdentifiers`: 系统自动隐藏的项目
- `visibleNavigationItemIdentifiers`: 当前可见的所有导航项

**关键方法**：
```objc
// 设置项目的用户隐藏状态
- (BOOL)setItemWithIdentifier:(NSString *)identifier isHiddenByUser:(BOOL)hidden;

// 设置项目的系统隐藏状态
- (BOOL)setItemWithIdentifier:(NSString *)identifier isHiddenBySystem:(BOOL)hidden;

// 检查项目是否被隐藏
- (BOOL)isItemWithIdentifierHidden:(NSString *)identifier;

// 更新导航视图模式
- (void)updateNavigationViewMode:(ENNoteCollectionWindowControllerNavigationViewMode)mode;
```

### 2. Navigation Outline View Module (树形导航视图)
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Outline View Module/`

#### 核心控制器
- **ENNavigationOutlineViewController**: 主控制器，管理整个树形导航视图
  - 继承自NSViewController
  - 实现ENNavigationOutlineViewItemOwner协议
  - 负责协调各个数据源和处理用户交互

- **ENNavigationOutlineView**: 自定义的NSOutlineView实现
  - 支持拖拽重排
  - 自定义右键菜单
  - 主题适配

#### 数据源系统 (Sources)
每个数据源负责管理特定类型的导航项：

1. **ENNavigationOutlineNotebooksSource**: 笔记本数据源
   - 管理个人笔记本和共享笔记本的完整层次结构
   - 支持笔记本堆栈(Stack)的分组管理
   - 处理链接笔记本和重复名称解决
   - 实时监听笔记本变化并动态更新展示

2. **ENNavigationOutlineTagsSource**: 标签数据源
   - 管理标签的树形结构
   - 支持标签的嵌套关系
   - 处理标签的搜索和过滤

3. **ENNavigationOutlineShortcutsSource**: 快捷方式数据源
   - 管理用户自定义的快捷方式
   - 支持拖拽重新排序
   - 动态更新快捷方式列表

4. **ENNavigationOutlineRecentNotesSource**: 最近笔记数据源
   - 显示最近访问的笔记
   - 自动更新列表
   - 支持时间排序

5. **ENNavigationOutlineTeamSpacesSource**: 团队空间数据源
   - 管理团队空间的导航
   - 处理团队权限和访问控制
   - 显示团队空间内的笔记本结构

6. **ENNavigationOutlineWorkspacesSource**: 工作空间数据源
   - 管理工作空间的切换
   - 处理工作空间特定的配置
   - 支持工作空间的创建和管理

7. **EverPenSource**: EverPen功能数据源
   - 管理手写笔记功能的完整导航结构
   - 处理智能笔记本和收藏夹的分类展示
   - 支持实时数据监听和动态更新
   - 集成手写笔记的层次化展示

#### 导航项类型 (Items)
位于`Items/`目录，包含各种类型的导航项：

- **ENNavigationOutlineViewModuleItem**: 模块项
  - 表示一个功能模块的导航项
  - 支持图标、标题、计数显示
  - 可配置选中状态和可见性

- **ENNavigationOutlineViewActionItem**: 操作项
  - 执行特定操作的导航项
  - 支持自定义操作处理
  - 可配置快捷键

- **ENNavigationOutlineViewGroupHeaderItem**: 分组标题项
  - 用于分组导航项的标题
  - 支持折叠/展开功能
  - 可自定义样式

- **ENNavigationOutlineViewAIRootItem**: AI功能根项
  - AI功能的入口点
  - 管理AI相关的子项
  - 支持AI功能的动态加载

- **ENNavigationOutlineViewUpgradeItem**: 升级项
  - 显示升级提示
  - 处理升级相关的操作
  - 支持不同的升级类型

- **ENNavigationOutlineDividerItem**: 分隔线项
  - 视觉分隔不同的导航区域
  - 支持自定义样式
  - 响应主题变化

### 3. Navigation Stack View Module (垂直堆叠导航视图)
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Stack View Module/`

#### 核心组件
- **ENNavigationStackViewController**: 堆叠视图控制器
  - 管理垂直堆叠的导航项
  - 处理项目的添加和移除
  - 支持选中状态管理

- **ENNavigationStackView**: 自定义的堆叠视图容器
  - 实现垂直布局逻辑
  - 支持动画过渡
  - 响应大小变化

- **ENNavigationNarrowViewController**: 窄模式视图控制器
  - 专门为窄窗口优化的视图
  - 简化的导航界面
  - 支持快速切换

#### 堆叠项组件 (Items)
- **ENNavigationStackViewBaseItem**: 所有堆叠项的基类
  - 定义通用的堆叠项接口
  - 处理基础的用户交互
  - 支持主题适配

- **ENNavigationStackViewModuleItem**: 模块堆叠项
  - 在堆叠视图中表示功能模块
  - 优化的紧凑布局
  - 支持图标和简短标题

- **ENNavigationStackViewActionItem**: 操作堆叠项
  - 堆叠视图中的操作项
  - 支持快速操作
  - 可配置的操作处理

- **NavigationStackViewBottomItem**: 底部固定项
  - 包含垃圾箱、帮助、自定义按钮
  - 始终位于导航底部
  - 支持右键菜单

- **ENNavigationStackViewUpgradeItem**: 升级提示项
  - 堆叠视图中的升级提示
  - 紧凑的升级信息展示
  - 支持快速升级操作

### 5. EverPen Module (手写笔记模块)
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Outline View Module/Sources/EverPenSource.swift`

#### 核心架构

**EverPenSource**是EverPen功能的数据源，负责管理手写笔记功能在侧边栏中的完整展示逻辑。

**核心特性**：
- **层次化结构**: 采用两级导航结构（堆栈项 → 笔记本项）
- **双分类系统**: 支持智能笔记本和收藏夹两种分类
- **实时监听**: 使用Monitor模式实时监听数据变化
- **动态更新**: 支持笔记本的增删改查和计数更新

#### 数据结构

```swift
// 数据源标识
static let identifier = "EverPEN"
static let objectIDStringPrefix = "ep-"

// 分类标签
static let notebookLabel = "Smart Notebook"      // 智能笔记本
static let archiveNotebookLabel = "Favorites Cabinet"  // 收藏夹

// 数据存储
private var notebookList = [PenNotebook]()       // 智能笔记本列表
private var archiveList = [PenNotebook]()        // 收藏夹列表
```

#### 层次结构

```
EverPEN (模块根节点)
├── Smart Notebook (智能笔记本堆栈)
│   ├── 笔记本A (pageCount: 5)
│   ├── 笔记本B (pageCount: 12)
│   └── ...
└── Favorites Cabinet (收藏夹堆栈)
    ├── 收藏笔记本C (pageCount: 3)
    ├── 收藏笔记本D (pageCount: 8)
    └── ...
```

#### 核心组件

**1. EverPENStackItem (堆栈项)**
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Outline View Module/Items/Object Items/EverPENStackItem.swift`

**职责**：
- 作为笔记本的分组容器（智能笔记本/收藏夹）
- 支持展开/折叠操作
- 处理分组级别的点击事件

**核心实现**：
```swift
class EverPENStackItem: ENNavigationOutlineViewObjectItem {
  // 堆栈项始终可展开
  override func isExpandable() -> Bool { return true }
  
  // 处理点击事件 - 展示对应的笔记本列表
  override func performClickAction(with owner: ENNavigationOutlineViewItemOwner) {
    if title == EverPenSource.notebookLabel {
      owner.noteCollectionWindowController.showPenNotebookList(forArchive: false)
    } else if title == EverPenSource.archiveNotebookLabel {
      owner.noteCollectionWindowController.showPenNotebookList(forArchive: true)
    }
  }
  
  // 双击事件 - 切换展开状态
  override func performDoubleClickAction(...) {
    owner.toggleNavigationOutlineViewItemExpanded(self)
  }
}
```

**2. EverPenNotebookItem (笔记本项)**
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Outline View Module/Items/Object Items/EverPenNotebookItem.swift`

**职责**：
- 表示单个EverPen笔记本
- 显示笔记本名称、图标和页面计数
- 处理笔记本的选择和打开操作

**核心实现**：
```swift
class EverPenNotebookItem: ENNavigationOutlineViewObjectItem {
  let notebook: PenNotebook          // 关联的笔记本对象
  let parentItem: EverPENStackItem   // 父级堆栈项引用
  
  // 显示笔记本名称
  override func label() -> String? { return notebook.name }
  
  // 根据类型显示不同图标
  override func image() -> NSImage? {
    return notebook.isArchiveNotebook ? 
      NSImage(named: "lnav-icon-pen-favorites") :    // 收藏夹图标
      NSImage(named: "lnav-icon-pen-notebook")       // 普通笔记本图标
  }
  
  // 显示页面计数
  override func count() -> Int { return notebook.pageCount }
  
  // 笔记本项不可展开
  override func isExpandable() -> Bool { return false }
  
  // 自定义单元格 - 显示小尺寸的内联计数
  override func setUpTableCellView(_ tableCellView: ENNavigationOutlineCellView, showingCount showCount: Bool) {
    super.setUpTableCellView(tableCellView, showingCount: true)
    let cell = tableCellView.textField?.cell as? ENNavigationOutlineViewTextFieldCell
    cell?.countStyle = .inlineSmall  // 使用小尺寸内联计数样式
  }
  
  // 处理点击事件 - 打开笔记本并记录分析事件
  override func performClickAction(with owner: ENNavigationOutlineViewItemOwner) {
    owner.noteCollectionWindowController.showPenNotebook(notebook)
    
    // 记录用户行为分析
    if notebook.isArchiveNotebook {
      owner.accountController.accountDecorator.analytics.recordLegacyEvent(
        withCategory: "nav_bar_wide", action: "click_favorites", label: "everpen")
    } else {
      owner.accountController.accountDecorator.analytics.recordLegacyEvent(
        withCategory: "nav_bar_wide", action: "click_notebook", label: "everpen")
    }
  }
}
```

#### 数据监听与更新机制

**实时监听系统**：
EverPenSource使用Monitor模式实现数据的实时监听和更新：

```swift
// 笔记本监听器
private var notebookMonitor: _NotebookListMonitor?
private var archiveNotebookMonitor: _ArchiveListMonitor?

// 添加监听器
private func addObservers() {
  // 智能笔记本变化监听
  notebookMonitor = _NotebookListMonitor { [weak self] stagedChangeset in
    guard let self = self else { return }
    let needReload = self.checkIfHasChanges(oldData: self.notebookList, stagedChangeset: stagedChangeset)
    if needReload {
      self.notebookList = stagedChangeset.last?.data ?? []
      self.owner?.sourceChanged(self)  // 通知控制器重新加载
    }
  }
  
  // 收藏夹变化监听
  archiveNotebookMonitor = _ArchiveListMonitor { [weak self] stagedChangeset in
    // 类似的变化检测逻辑
  }
}
```

**变化检测算法**：
```swift
private func checkIfHasChanges(oldData: [PenNotebook], stagedChangeset: PenStagedChangeset<PenNotebook>) -> Bool {
  var hasChanges = false
  for changeset in stagedChangeset {
    for change in changeset.changes {
      if case .updated(let indexPath) = change, indexPath.item < oldData.count {
        let new = changeset.data[indexPath.item]
        let old = oldData[indexPath.item]
        // 检测关键属性变化：GUID、名称、页面计数
        if new.guid != old.guid || new.name != old.name || new.pageCount != old.pageCount {
          hasChanges = true
          break
        }
      } else {
        hasChanges = true  // 新增或删除操作
        break
      }
    }
  }
  return hasChanges
}
```

#### 重载机制

**reload()方法**负责构建完整的EverPen导航结构：

```swift
override func reload() -> UInt {
  guard let owner = owner else { return 0 }
  removeAndTearDownAllChildren()  // 清理旧的子项
  
  // 创建两个主要的堆栈项
  let notebookStackItem = EverPENStackItem(name: Self.notebookLabel, ...)      // 智能笔记本
  let archiveNotebookStackItem = EverPENStackItem(name: Self.archiveNotebookLabel, ...) // 收藏夹
  
  // 注册到项目映射表
  itemMap.setObject(notebookStackItem, forKey: Self.notebookLabel as NSString)
  itemMap.setObject(archiveNotebookStackItem, forKey: Self.archiveNotebookLabel as NSString)
  
  // 为每个智能笔记本创建子项
  notebookList.forEach { notebook in
    let item = EverPenNotebookItem(notebook: notebook, parentItem: notebookStackItem, ...)
    notebookStackItem.addChild(item)
    self.itemMap.setObject(item, forKey: (notebook.guid as NSString))
  }
  
  // 为每个收藏夹创建子项
  archiveList.forEach { notebook in
    let item = EverPenNotebookItem(notebook: notebook, parentItem: archiveNotebookStackItem, ...)
    archiveNotebookStackItem.addChild(item)
    self.itemMap.setObject(item, forKey: (notebook.guid as NSString))
  }
  
  // 添加到模块项
  moduleItem.addChild(notebookStackItem)
  moduleItem.addChild(archiveNotebookStackItem)
  
  return UInt(notebookList.count + archiveList.count)  // 返回总项目数
}
```

#### 用户交互流程

**1. 堆栈项交互**：
```
用户点击"Smart Notebook" → EverPENStackItem.performClickAction() 
→ showPenNotebookList(forArchive: false) → 显示智能笔记本列表视图
```

**2. 笔记本项交互**：
```
用户点击具体笔记本 → EverPenNotebookItem.performClickAction() 
→ showPenNotebook(notebook) → 打开对应的笔记本内容
→ 记录分析事件 → 更新用户行为数据
```

**3. 展开/折叠**：
```
用户双击堆栈项 → EverPENStackItem.performDoubleClickAction() 
→ toggleNavigationOutlineViewItemExpanded() → 切换子项显示状态
```

#### 集成特性

**1. 位置定义**：
- 使用`ENNavigationOutlineViewItemLocation.everPEN`作为位置标识
- 在导航层次中有固定的显示位置

**2. 对象ID管理**：
- 使用"ep-"前缀标识EverPen相关对象
- 支持通过GUID快速查找笔记本项

**3. 分析集成**：
- 完整的用户行为追踪
- 区分普通笔记本和收藏夹的点击事件
- 支持A/B测试和功能优化分析

**4. 主题适配**：
- 图标和文字颜色自动适配系统主题
- 支持深色/浅色模式切换
- 计数显示样式响应主题变化

这个EverPen模块展示了一个完整的功能模块如何集成到Evernote的侧边栏系统中，包括数据管理、UI展示、用户交互和系统集成的各个方面。

### 6. Notebooks Module (笔记本模块)
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Outline View Module/Sources/ENNavigationOutlineNotebooksSource.m`

#### 核心架构

**ENNavigationOutlineNotebooksSource**是笔记本功能的数据源，负责管理Evernote笔记本在侧边栏中的完整展示逻辑，是系统中最复杂的数据源之一。

**核心特性**：
- **三级层次结构**: 支持模块项 → 堆栈项 → 笔记本项的完整层次
- **堆栈分组**: 自动根据笔记本堆栈进行分组管理
- **双重持久化监听**: 同时监听笔记本和链接笔记本的变化
- **重名处理**: 智能处理共享笔记本的重名显示问题
- **拖拽支持**: 完整的拖拽操作支持，包括工作空间移动

#### 数据结构

```objc
// 数据源标识
static NSString *identifier = @"notebooks"

// 核心属性
@property NSMapTable<NSString *, ENNavigationOutlineViewStackItem *> *stackToItemMap;  // 堆栈映射表
@property BOOL needsModuleOutlineViewItem;      // 是否需要模块项而非组标题项
@property BOOL shouldLoadNotebookList;          // 是否应该加载笔记本列表

// 位置定义
ENNavigationOutlineViewItemLocationNotebooks    // 笔记本模块位置标识
```

#### 层次结构

```
Notebooks (模块根节点)
├── [无堆栈的笔记本]
│   ├── 个人笔记本A
│   ├── 个人笔记本B
│   └── ...
├── 堆栈名称A (堆栈分组)
│   ├── 笔记本C (属于堆栈A)
│   ├── 笔记本D (属于堆栈A)
│   └── ...
├── 堆栈名称B (堆栈分组)
│   ├── 笔记本E (属于堆栈B)
│   └── ...
└── [列表底部边距项]
```

#### 核心组件

**1. ENNavigationOutlineViewModuleItem (模块项)**
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Outline View Module/Items/ENNavigationOutlineViewModuleItem.m`

**职责**：
- 作为笔记本模块的根容器
- 动态计算和显示笔记本总数
- 支持自定义外观（小型/大型模式）
- 处理模块级别的拖拽操作

**核心实现**：
```objc
@interface ENNavigationOutlineViewModuleItem : ENNavigationOutlineViewParentItem
@property (copy, nonnull, readonly) NSString *moduleIdentifier;  // 模块标识符
@property BOOL smallAppearance;                                  // 小型外观模式
@property (strong, nullable) NSImage *customImage;              // 自定义图标
@property (weak, nullable) id<ENNavigationOutlineViewModuleItemDelegate> delegate;  // 代理

// 动态计数系统
@property (copy, nullable) NSArray<NSString *> *observedCountKeyPaths;   // 监听的计数路径
@property (copy, nullable) NSString *observedNameKeyPath;                // 监听的名称路径
@end

// 动态计数计算 - 通过KVO监听多个键路径
- (NSInteger)count {
  NSInteger count = 0;
  for (NSString *keyPath in self.observedCountKeyPaths) {
    NSNumber *value = [self valueForKeyPath:keyPath];
    if (!value) return -1;
    NSInteger intVal = [value integerValue];
    if (intVal > -1) {
      count += intVal;
    } else {
      return -1;
    }
  }
  return count;
}

// 点击处理 - 显示模块并根据需要重置状态
- (void)performClickActionWithOwner:(id<ENNavigationOutlineViewItemOwner>)owner {
  BOOL resetViewState = [[self moduleIdentifier] isEqualToString:owner.noteCollectionWindowController.currentModuleIdentifier];
  [owner.noteCollectionWindowController showModuleWithIdentifier:[self moduleIdentifier] resetViewState:resetViewState sender:self];
}
```

**2. ENNavigationOutlineViewStackItem (堆栈项)**
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Outline View Module/Items/Object Items/ENNavigationOutlineViewStackItem.h`

**职责**：
- 表示笔记本堆栈分组
- 作为堆栈内笔记本的容器
- 支持堆栈的展开/折叠操作

**核心实现**：
```objc
@interface ENNavigationOutlineViewStackItem : ENNavigationOutlineViewObjectItem
- (nonnull instancetype)initWithName:(nonnull NSString *)name
                            location:(ENNavigationOutlineViewItemLocation)location
                          showsImage:(BOOL)showsImage
                       keyEquivalent:(nullable NSString *)keyEquivalent
                               owner:(nonnull id<ENNavigationOutlineViewItemOwner>)owner;

@property (copy, nullable, readonly) NSString *persistenceKey;  // 持久化键，用于区分业务/个人笔记本
@end
```

**3. ENNavigationOutlineViewNotebookItem (笔记本项)**
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Navigation Outline View Module/Items/Object Items/ENNavigationOutlineViewNotebookItem.h`

**职责**：
- 表示单个笔记本
- 显示笔记本名称、图标和笔记计数
- 处理重名笔记本的特殊显示逻辑

**核心实现**：
```objc
@interface ENNavigationOutlineViewNotebookItem : ENNavigationOutlineViewObjectItem
@property (readonly) ENNotebookMO *notebook;                             // 关联的笔记本对象
@property (nonatomic) BOOL showNotebookNameWithOwnerIfNotOwnedBySelf;    // 是否显示所有者名称
@end
```

#### 重载机制与堆栈管理

**reload()方法**是笔记本模块的核心，负责构建完整的层次结构：

```objc
- (NSUInteger)reload {
  [self removeAndTearDownAllChildren];                    // 清理现有结构
  [self removeAndTearDownAllMapTableObjects:self.stackToItemMap];  // 清理堆栈映射
  
  if (self.shouldLoadNotebookList) {
    NSMutableDictionary<NSString *, ENNavigationOutlineViewStackItem *> *stackItems = [NSMutableDictionary dictionary];
    
    // 获取所有笔记本
    NSArray<ENNotebookMO *> *allNotebooks = self.owner.accountController.account.notebooksProtocol.allNotebooks;
    
    NSUInteger index = 0;
    for (ENNotebookMO *notebook in allNotebooks) {
      // 创建笔记本项
      ENNavigationOutlineViewNotebookItem *notebookItem = 
        [[ENNavigationOutlineViewNotebookItem alloc] initWithObject:notebook
                                                           location:[[self class] location]
                                                         showsImage:self.showsChildImage
                                                      keyEquivalent:[self makeKeyEquivalent:index++]
                                                              owner:self.owner];
      
      ENNavigationOutlineViewParentItem *parentItem = notebooksModuleItem;
      
      // 堆栈处理逻辑
      NSString *stack = [notebook displayStack];
      if (stack) {
        // 如果笔记本属于堆栈，创建或获取堆栈项
        ENNavigationOutlineViewStackItem *stackItem = stackItems[stack];
        if (!stackItem) {
          stackItem = [[ENNavigationOutlineViewStackItem alloc] initWithName:stack
                                                                    location:[[self class] location]
                                                                  showsImage:self.showsChildImage
                                                               keyEquivalent:nil
                                                                       owner:self.owner];
          stackItems[stack] = stackItem;
          [parentItem addChild:stackItem];
          [self.stackToItemMap setObject:stackItem forKey:stackItem.persistenceKey];
        }
        [stackItem addChild:notebookItem];  // 添加到堆栈
      } else {
        [parentItem addChild:notebookItem]; // 直接添加到模块
      }
      [self.itemMap setObject:notebookItem forKey:notebook];
    }
    
    // 排序 - 按标签字母顺序排序
    NSSortDescriptor *labelSort = [[NSSortDescriptor alloc] initWithKey:@"label" ascending:YES selector:@selector(localizedStandardCompare:)];
    [notebooksModuleItem sortChildrenUsingDescriptors:@[labelSort] recursive:YES];
  }
  
  // 处理重名笔记本
  [self handleDuplicateNotebookNames];
  
  // 添加底部边距
  [self addListBottomMarginItem];
  
  return [allNotebooks count];
}
```

#### 重名笔记本处理

Evernote支持共享笔记本，可能出现多个笔记本同名的情况。系统通过智能算法处理这种情况：

```objc
// 步骤1：按笔记本名称分组
NSMutableDictionary *notebookItemsKeyedByNotebookName = [NSMutableDictionary new];
for (ENNavigationOutlineViewNotebookItem *notebookItem in self.itemMap.objectEnumerator) {
  NSString *notebookName = notebookItem.notebook.displayName.lowercaseString;
  if (!notebookName.length) continue;
  
  NSMutableArray *notebooks = notebookItemsKeyedByNotebookName[notebookName];
  if (notebooks) {
    [notebooks addObject:notebookItem];
  } else {
    notebookItemsKeyedByNotebookName[notebookName] = [NSMutableArray arrayWithObject:notebookItem];
  }
}

// 步骤2：为重名笔记本设置显示所有者名称
for (NSArray *notebookItemsWithMatchingName in notebookItemsKeyedByNotebookName.allValues) {
  BOOL hasDuplicateName = (notebookItemsWithMatchingName.count > 1);
  for (ENNavigationOutlineViewNotebookItem *notebookItem in notebookItemsWithMatchingName) {
    notebookItem.showNotebookNameWithOwnerIfNotOwnedBySelf = hasDuplicateName;  // 重名时显示所有者
  }
}
```

#### 实时监听与更新机制

**双重持久化监听**：笔记本模块同时监听普通笔记本和链接笔记本的变化：

```objc
// 初始化时添加监听器
- (instancetype)initWithNavigationViewItemOwner:(id<ENNavigationOutlineViewItemOwner>)owner ... {
  self = [super init...];
  if (self) {
    _stackToItemMap = [NSMapTable strongToStrongObjectsMapTable];
    [self.accountController.account.notebooksProtocol addNotebookObserver:self];
    [self.accountController.account.linkedNotebooksProtocol addLinkedNotebookObserver:self];
  }
  return self;
}

// 笔记本变化处理
- (void)notebooksDidChange:(ENNotebookChanges *)changesSinceLastUpdate inAccount:(id<ENEvernoteAccountProtocol>)account {
  if (!self.shouldLoadNotebookList) return;  // 如果不需要加载笔记本列表则忽略
  
  BOOL needToReloadNotebookList = NO;
  
  if (changesSinceLastUpdate.creations.count || changesSinceLastUpdate.deletions.count) {
    // 有笔记本增加或删除，需要重新加载
    needToReloadNotebookList = YES;
  } else {
    // 检查更新是否影响显示
    for (ENNotebookUpdate *update in changesSinceLastUpdate.updates) {
      ENNotebookMO *notebook = [self.owner.accountController.account.notebooksProtocol notebookWithID:update.objectID];
      if (notebook) {
        ENNavigationOutlineViewItem *item = [self itemForObject:notebook];
        if (!item) {
          AppLogError(@"%@ %@ has no left nav item", [notebook class], notebook.displayName);
        }
        if ([item representedObjectHasChanged]) {
          needToReloadNotebookList = YES;
          break;
        }
      }
    }
  }
  
  if (needToReloadNotebookList) {
    [self notifyOwnerAboutChange];  // 通知控制器重新加载
  }
}

// 链接笔记本变化处理 - 类似逻辑
- (void)linkedNotebooksDidChange:(ENLinkedNotebookChanges *)changesSinceLastUpdate inAccount:(id<ENEvernoteAccountProtocol>)account {
  // 类似的变化检测和处理逻辑
}
```

#### 拖拽操作支持

笔记本模块支持完整的拖拽操作，通过代理模式实现：

```objc
// 模块项代理协议
@protocol ENNavigationOutlineViewModuleItemDelegate <NSObject>
@optional
- (BOOL)navigationOutlineViewModuleItem:(ENNavigationOutlineViewModuleItem *)item canAcceptDropOfNotebook:(ENNotebookMO *)notebook;
- (BOOL)navigationOutlineViewModuleItem:(ENNavigationOutlineViewModuleItem *)item acceptsPasteboardTypeDrops:(NSArray<NSString *> *)pasteboardTypes;
- (BOOL)navigationOutlineViewModuleItem:(ENNavigationOutlineViewModuleItem *)item acceptDrop:(id<NSDraggingInfo>)info forAccountController:(ENAccountController *)accountController;
@end

// 实现拖拽处理
- (BOOL)navigationOutlineViewModuleItem:(ENNavigationOutlineViewModuleItem *)item canAcceptDropOfNotebook:(ENNotebookMO *)notebook {
  return [ENWorkspaceMO canMoveNotebooks:@[notebook] toWorkspace:nil error:nil];  // 检查是否可以移动到工作空间
}

- (BOOL)navigationOutlineViewModuleItem:(ENNavigationOutlineViewModuleItem *)item acceptDrop:(id<NSDraggingInfo>)info forAccountController:(ENAccountController *)accountController {
  NSArray *notebooks = [accountController.accountDecorator notebooksFromPasteboard:info.draggingPasteboard];
  if (!notebooks) return NO;
  
  [accountController.moveHelper moveNotebooks:notebooks toWorkspace:nil];  // 执行移动操作
  return YES;
}
```

#### 配置与外观管理

**双模式支持**：笔记本模块支持两种显示模式：

```objc
- (ENNavigationOutlineViewParentItem *)makeOutlineViewItem {
  if (self.needsModuleOutlineViewItem) {
    // 模块项模式 - 带图标和计数的完整模块显示
    ENNavigationOutlineViewModuleItem *moduleItem = 
      [[ENNavigationOutlineViewModuleItem alloc] initWithModuleIdentifier:[EN4NotebooksNABModule identifier] owner:self.owner];
    moduleItem.delegate = self;
    return moduleItem;
  } else {
    // 组标题模式 - 简单的文本标题显示
    return [[ENNavigationOutlineViewGroupHeaderItem alloc] initWithIdentifier:[[self class] identifier]
                                                                        label:NSLocalizedString(@"Notebooks", @"Notebooks left nav header (title case)")
                                                                        owner:self.owner];
  }
}
```

**外观自适应**：
```objc
// 模块项支持小型和大型外观
- (CGFloat)rowHeightForCellWidth:(CGFloat)cellWidth {
  if (self.smallAppearance) {
    return kENNavigationOutlineViewGroupHeaderCellHeight;   // 小型高度
  } else {
    return kENNavigationOutlineViewLargeCellHeight;         // 大型高度
  }
}

- (NSString *)viewIdentifier {
  if (self.smallAppearance) {
    return kENNavigationOutlineViewGroupCell;               // 小型单元格
  } else {
    return kENNavigationOutlineViewLargeCell;               // 大型单元格
  }
}
```

#### 集成特性

**1. 工作空间集成**：
- 支持笔记本在工作空间间的移动
- 智能检测移动权限
- 集成拖拽操作处理器

**2. 搜索集成**：
- 支持快速键盘导航
- 自动生成键盘快捷键
- 标签排序和搜索优化

**3. 持久化管理**：
- 堆栈状态持久化
- 展开/折叠状态记忆
- 用户偏好设置保存

**4. 错误处理**：
- 完善的日志记录系统
- 异常状态检测和恢复
- 数据一致性验证

这个Notebooks模块展示了Evernote侧边栏系统中最复杂的数据源实现，涵盖了层次化数据管理、实时监听、拖拽操作、重名处理等多个高级特性，为用户提供了完整而流畅的笔记本管理体验。

### 4. Menu System (菜单系统)
`Source/User-Interface/Note-Windows-Module/Note Collection Window/SideNavigation/Menu/`

#### 上下文菜单
- **NavigationContextMenuViewController**: 自定义导航上下文菜单
  - 提供丰富的上下文操作
  - 支持动态菜单项
  - 主题适配的菜单样式

- **NavigationContextMenuWindowController**: 菜单窗口控制器
  - 管理菜单窗口的显示
  - 处理菜单的定位和大小
  - 支持键盘导航

- **NavigationCustomMenuItem**: 自定义菜单项
  - 扩展标准菜单项功能
  - 支持自定义图标和样式
  - 可配置的操作处理

- **MoreNoteMenuHelper**: 更多选项菜单辅助类
  - 处理"更多"菜单的逻辑
  - 管理隐藏项目的显示
  - 支持动态菜单生成

## 核心交互流程

### 1. 导航项显示逻辑
```
ENNavigationManager 
    ↓ 决定可见性
ENNavigationOutlineViewController/ENNavigationStackViewController
    ↓ 根据模式渲染
Data Sources → Navigation Items → UI Display
```

**详细流程**：
1. ENNavigationManager根据用户设置和系统状态确定哪些项目应该显示
2. 相应的ViewController根据当前模式选择合适的渲染方式
3. Data Sources提供具体的数据和状态信息
4. Navigation Items负责实际的UI渲染和用户交互

### 2. 用户交互处理
```
用户点击/右键 → Navigation Controller → 相应的Source处理 → 更新选中状态 → 通知主窗口控制器
```

**交互类型**：
- **单击**: 选择导航项，更新主内容区域
- **右键**: 显示上下文菜单，提供相关操作
- **拖拽**: 重新排序（适用于快捷方式等）
- **双击**: 执行默认操作（如重命名）

### 3. 模式切换机制
```
ENNavigationManager.updateNavigationViewMode()
    ↓
ENNoteCollectionWindowController 协调切换
    ↓  
显示对应的 Outline View 或 Stack View
```

**切换触发条件**：
- 窗口大小变化
- 用户手动切换
- 系统建议切换（基于可用空间）

## 主要特性

### 1. 自适应布局
- **响应式设计**: 根据窗口大小自动调整布局
- **智能切换**: 窄模式下自动切换到堆叠视图
- **动态缩放**: 图标和文字根据可用空间调整大小
- **流畅动画**: 布局变化时提供平滑的过渡动画

### 2. 可定制性
- **用户偏好**: 用户可以隐藏/显示特定的导航项
- **自定义排序**: 支持拖拽重新排序（快捷方式等）
- **主题适配**: 完整的深色/浅色模式支持
- **个性化**: 记住用户的导航偏好和使用习惯

### 3. 状态管理
- **KVO监听**: 使用Key-Value Observing监听各种状态变化
- **持久化**: 用户的导航偏好设置会被保存
- **实时更新**: 计数和状态指示器实时更新
- **状态同步**: 多个视图之间的状态保持同步

### 4. AI功能集成
- **专门入口**: 专门的AI功能导航项
- **快速访问**: AI写作和AI阅读的快速访问
- **动态显示**: 根据AI功能可用性动态显示相关项目
- **智能建议**: 基于用户行为提供智能导航建议

## 关键设计模式

1. **MVC架构**: 每个组件都遵循Model-View-Controller模式
   - Model: 数据源(Sources)负责数据管理
   - View: 导航项(Items)负责UI显示
   - Controller: 视图控制器负责协调逻辑

2. **数据源模式**: 使用专门的Source类管理数据
   - 分离数据逻辑和展示逻辑
   - 支持数据的动态更新
   - 便于单元测试和维护

3. **观察者模式**: 大量使用KVO进行状态同步
   - 自动响应数据变化
   - 减少手动同步的复杂性
   - 提高系统的响应性

4. **策略模式**: 不同的导航模式使用不同的渲染策略
   - Outline View策略用于树形导航
   - Stack View策略用于堆叠导航
   - 便于扩展新的导航模式

5. **组合模式**: 导航项的层次结构管理
   - 统一处理单个项目和项目组
   - 支持嵌套的导航结构
   - 简化复杂层次的管理

## 扩展性考虑

### 1. 新功能集成
- 通过创建新的Source类轻松添加新的数据源
- 通过创建新的Item类添加新的导航项类型
- 最小化对现有代码的影响

### 2. 平台扩展
- 清晰的架构便于向其他平台移植
- 分离的数据层和UI层支持不同的UI框架
- 标准化的接口设计

### 3. 性能优化
- 懒加载机制减少初始化时间
- 虚拟化支持大量导航项
- 高效的状态管理减少不必要的更新

这个架构设计充分体现了模块化、可扩展性和用户体验的平衡，为Evernote的复杂功能提供了清晰直观的导航界面。通过精心设计的组件分离和清晰的职责划分，这个系统既能满足当前的功能需求，也为未来的扩展留下了充足的空间。