# Evernote Mac App Editor Area 右侧添加 AI Area 架构设计文档

## 概述

本文档详细描述了在 Evernote Mac 客户端 Editor Area 右侧新增 AI Area 的完整架构设计方案。新的 AI Area 将与现有的 Editor Area 通过可拖拽的分割视图共存，用户可以自由调整两个区域的大小比例，实现编辑器与 AI 功能的无缝集成。

## 设计目标

- **无缝集成**: 在不破坏现有架构的基础上添加 AI 功能
- **用户友好**: 提供直观的拖拽调整和快速切换功能
- **模块化设计**: AI Area 作为独立模块，便于维护和扩展
- **性能优化**: 采用懒加载和合理的资源管理策略
- **状态持久化**: 保存和恢复用户的布局偏好设置

## 新架构整体视图

```
┌─────────────────────────────────────────────────────────────────────────────────────┐
│                    ENNoteCollectionWindowController (主窗口)                          │
├─────────────────────────────────────────────────────────────────────────────────────┤
│                          ENSplitViewController                                        │
│                               ↓                                                     │
│  ┌─────────────────────┬─────────────────────────────────────────────────────────┐  │
│  │                     │                                                       │  │
│  │  navigationView     │              contentView                              │  │
│  │   (左侧分割面)        │              (右侧分割面)                              │  │
│  │                     │                                                       │  │
│  │ ENNavigation        │               ENViewModulesManager                    │  │
│  │ OutlineView         │                        ↓                              │  │
│  │ Controller          │              EN4NotesModule (默认模块)                  │  │
│  │                     │                        ↓                              │  │
│  │ ├── 📓 All Notes    │  ┌─────────────────────┬─────────────────────────────┐  │  │
│  │ ├── 📒 Notebooks    │  │   Notes List Area   │   Main Content Area        │  │  │
│  │ ├── 🏷️  Tags        │  │   (笔记列表区域)     │   (主内容区域)              │  │  │
│  │ ├── 🔗 Shortcuts    │  │                     │  ⬅ 新增 SplitView ➡       │  │  │
│  │ ├── ⏰ Recent Notes │  │ ┌─────────────────┐ │ ┌─────────────┬─────────────┐ │  │  │
│  │ ├── 👥 Team Spaces  │  │ │ Header Container│ │ │ Editor Area │  AI Area    │ │  │  │
│  │ ├── 🗂️  Workspaces  │  │ │ (标题/搜索区域)  │ │ │             │             │ │  │  │
│  │ ├── 🤖 AI Features  │  │ └─────────────────┘ │ │ ┌─────────┐ │ ┌─────────┐ │ │  │  │
│  │ ├── 💬 Messages     │  │ ┌─────────────────┐ │ │ │Toolbar  │ │ │AI Header│ │ │  │  │
│  │ ├── 📚 Library      │  │ │ Notes Container │ │ │ ├─────────┤ │ ├─────────┤ │ │  │  │
│  │ └── 🗑️  Trash       │  │ │ ┌─────────────┐ │ │ │ Content │ │ │AI Content│ │ │  │  │
│  │                     │  │ │ │ Note Item 1 │ │ │ │ WebView │ │ │         │ │ │  │  │
│  │                     │  │ │ ├─────────────┤ │ │ │         │ │ │ • Chat  │ │ │  │  │
│  │                     │  │ │ │ Note Item 2 │ │ │ │         │ │ │ • Tools │ │ │  │  │
│  │                     │  │ │ └─────────────┘ │ │ ├─────────┤ │ │ • Hist. │ │ │  │  │
│  │                     │  │ └─────────────────┘ │ │Tags Area│ │ └─────────┘ │ │  │  │
│  │                     │  └─────────────────────┘ └─────────────┴─────────────┘ │  │  │
│  │                     │                                                       │  │  │
│  └─────────────────────┴─────────────────────────────────────────────────────────┘  │
│                                                                                     │
└─────────────────────────────────────────────────────────────────────────────────────┘
```

## 一、架构改动概览

### 1.1 核心改动点

1. **EN4NotesModule 扩展**: 添加对 AI Area 的支持和管理
2. **新增容器控制器**: 创建 `ENEditorAIContainerViewController` 管理编辑器和 AI 区域
3. **AI Area 组件**: 实现完整的 AI 功能模块
4. **分割视图集成**: 通过 `NSSplitView` 实现可拖拽的布局调整
5. **工具栏集成**: 在现有工具栏中添加 AI 切换功能

### 1.2 新增组件清单

| 组件名称 | 文件位置 | 主要职责 |
|----------|----------|----------|
| `ENEditorAIContainerViewController` | `Note-Editor/Container/` | 管理编辑器和AI区域的分割布局 |
| `ENAIAreaViewController` | `AI-Area/` | AI 区域主控制器 |
| `ENAIHeaderViewController` | `AI-Area/Header/` | AI 区域头部控制 |
| `ENAIChatViewController` | `AI-Area/Chat/` | AI 聊天功能 |
| `ENAIToolsViewController` | `AI-Area/Tools/` | AI 工具集合 |
| `ENAIHistoryViewController` | `AI-Area/History/` | AI 交互历史 |

## 二、详细视图层级结构

### 2.1 新的视图层级关系

```
EN4NotesModule.view
├── noteListStackView (左侧 - 笔记列表区域) [保持不变]
│   ├── Header Container (pqF-d0-YTm)
│   ├── Assisted Search Container (0KN-Gp-b73)
│   ├── Privacy Lock Tips Container (KDY-CU-0ia)
│   ├── CommEngine Card Container (d9C-mg-Ofo)
│   ├── Depth Search Container (IsN-9u-fmn)
│   └── Notes Container (Lst-Cf-qmL)
└── editorAIContainerView (右侧 - 编辑器+AI容器) [新增]
    └── ENEditorAIContainerViewController.view
        └── editorAISplitView (水平分割视图)
            ├── editorContainerView (左侧 - 编辑器区域)
            │   └── ENNoteEditorViewController.view [现有组件]
            │       ├── toolbarContainerView (zUU-q9-eh3)
            │       │   └── SuperNoteEditorToolbarViewController
            │       ├── noteContentView (tBS-pg-TLO)
            │       │   ├── loadingContainerView (2036)
            │       │   ├── webViewContainer (10)
            │       │   │   └── 动态编辑器 WebView
            │       │   └── infoAndFormattingBarContainer
            │       │       └── ENUnoEditorFormattingBar
            │       └── noteTagContainer (xwi-z9-Qqx)
            │           └── ENNoteTagsFieldController
            └── aiAreaContainerView (右侧 - AI区域) [新增]
                └── ENAIAreaViewController.view
                    ├── aiHeaderView
                    │   └── ENAIHeaderViewController
                    ├── aiContentView
                    │   ├── ENAIChatViewController.view
                    │   ├── ENAIToolsViewController.view
                    │   └── ENAIHistoryViewController.view
                    └── aiFooterView
```

### 2.2 尺寸和约束规范

```objc
// 尺寸约束定义
#define EN_EDITOR_MIN_WIDTH     400.0f    // 编辑器最小宽度
#define EN_AI_AREA_MIN_WIDTH    200.0f    // AI区域最小宽度
#define EN_AI_AREA_DEFAULT_WIDTH 280.0f   // AI区域默认宽度
#define EN_AI_AREA_MAX_WIDTH    400.0f    // AI区域最大宽度

// 偏好设置键
extern NSString * const ENAIAreaVisibleKey;        // "ENAIAreaVisible"
extern NSString * const ENAIAreaWidthKey;          // "ENAIAreaWidth"
extern NSString * const ENAIAreaLastModeKey;       // "ENAIAreaLastMode"
```

## 三、核心组件设计

### 3.1 ENEditorAIContainerViewController - 容器控制器

#### 3.1.1 类定义

```objc
@interface ENEditorAIContainerViewController : NSViewController <NSSplitViewDelegate>

// 核心属性
@property (strong, readonly) ENNoteEditorViewController *editorViewController;
@property (strong, readonly) ENAIAreaViewController *aiAreaViewController;
@property (weak, readonly) ENAccountController *accountController;

// XIB 连接
@property (weak) IBOutlet NSSplitView *editorAISplitView;
@property (weak) IBOutlet NSView *editorContainerView;
@property (weak) IBOutlet NSView *aiAreaContainerView;

// 状态管理
@property (assign) BOOL isAIAreaVisible;
@property (assign) CGFloat aiAreaWidth;

// 初始化方法
- (instancetype)initWithEditorViewController:(ENNoteEditorViewController *)editorViewController
                           accountController:(ENAccountController *)accountController;

// 控制方法
- (void)showAIArea:(BOOL)show animated:(BOOL)animated;
- (void)toggleAIArea;
- (void)setAIAreaWidth:(CGFloat)width animated:(BOOL)animated;

// 状态持久化
- (void)saveLayoutState;
- (void)restoreLayoutState;

@end
```

#### 3.1.2 分割视图委托实现

```objc
@implementation ENEditorAIContainerViewController

#pragma mark - NSSplitViewDelegate

- (BOOL)splitView:(NSSplitView *)splitView canCollapseSubview:(NSView *)subview {
    // 只有 AI 区域可以折叠
    return (subview == self.aiAreaContainerView);
}

- (CGFloat)splitView:(NSSplitView *)splitView 
constrainMinCoordinate:(CGFloat)proposedMinimumPosition 
         ofSubviewAt:(NSInteger)dividerIndex {
    // 确保编辑器最小宽度
    return EN_EDITOR_MIN_WIDTH;
}

- (CGFloat)splitView:(NSSplitView *)splitView 
constrainMaxCoordinate:(CGFloat)proposedMaximumPosition 
         ofSubviewAt:(NSInteger)dividerIndex {
    // 确保 AI 区域最小宽度
    CGFloat totalWidth = splitView.frame.size.width;
    return totalWidth - EN_AI_AREA_MIN_WIDTH;
}

- (void)splitViewDidResizeSubviews:(NSNotification *)notification {
    // 实时保存 AI 区域宽度
    if (self.isAIAreaVisible) {
        self.aiAreaWidth = self.aiAreaContainerView.frame.size.width;
        [self saveLayoutState];
    }
    
    // 通知其他组件布局发生变化
    [[NSNotificationCenter defaultCenter] postNotificationName:ENEditorAILayoutDidChangeNotification 
                                                        object:self];
}

- (BOOL)splitView:(NSSplitView *)splitView shouldAdjustSizeOfSubview:(NSView *)view {
    // 编辑器区域优先调整大小
    return (view == self.editorContainerView);
}

@end
```

#### 3.1.3 XIB 文件结构

**ENEditorAIContainerViewController.xib**:
```xml
<?xml version="1.0" encoding="UTF-8"?>
<document type="com.apple.InterfaceBuilder3.Cocoa.XIB" version="3.0">
    <objects>
        <customObject id="-2" userLabel="File's Owner" customClass="ENEditorAIContainerViewController">
            <connections>
                <outlet property="editorAISplitView" destination="splitView" id="splitView-connection"/>
                <outlet property="editorContainerView" destination="editorContainer" id="editor-connection"/>
                <outlet property="aiAreaContainerView" destination="aiContainer" id="ai-connection"/>
                <outlet property="view" destination="containerView" id="view-connection"/>
            </connections>
        </customObject>
        
        <customView id="containerView" userLabel="Container View">
            <rect key="frame" x="0.0" y="0.0" width="700" height="500"/>
            <autoresizingMask key="autoresizingMask" widthSizable="YES" heightSizable="YES"/>
            <subviews>
                <splitView id="splitView" dividerStyle="thin" vertical="YES" 
                          translatesAutoresizingMaskIntoConstraints="NO">
                    <rect key="frame" x="0.0" y="0.0" width="700" height="500"/>
                    <subviews>
                        <!-- 编辑器容器 -->
                        <customView id="editorContainer" identifier="editorContainer">
                            <rect key="frame" x="0" y="0" width="420" height="500"/>
                            <autoresizingMask key="autoresizingMask"/>
                        </customView>
                        
                        <!-- AI 区域容器 -->
                        <customView id="aiContainer" identifier="aiContainer">
                            <rect key="frame" x="421" y="0" width="280" height="500"/>
                            <autoresizingMask key="autoresizingMask"/>
                        </customView>
                    </subviews>
                    <holdingPriorities>
                        <real value="250"/>  <!-- 编辑器优先级 -->
                        <real value="200"/>  <!-- AI 区域优先级 -->
                    </holdingPriorities>
                    <connections>
                        <outlet property="delegate" destination="-2" id="delegate-connection"/>
                    </connections>
                </splitView>
            </subviews>
            
            <!-- 自动布局约束 -->
            <constraints>
                <constraint firstItem="splitView" firstAttribute="top" 
                           secondItem="containerView" secondAttribute="top" id="top-constraint"/>
                <constraint firstItem="splitView" firstAttribute="leading" 
                           secondItem="containerView" secondAttribute="leading" id="leading-constraint"/>
                <constraint firstItem="splitView" firstAttribute="trailing" 
                           secondItem="containerView" secondAttribute="trailing" id="trailing-constraint"/>
                <constraint firstItem="splitView" firstAttribute="bottom" 
                           secondItem="containerView" secondAttribute="bottom" id="bottom-constraint"/>
            </constraints>
        </customView>
    </objects>
</document>
```

### 3.2 ENAIAreaViewController - AI 区域主控制器

#### 3.2.1 类定义和模式枚举

```objc
// AI 区域模式枚举
typedef NS_ENUM(NSUInteger, ENAIAreaMode) {
    ENAIAreaModeChat = 0,     // 聊天模式 - AI对话交互
    ENAIAreaModeTools,        // 工具模式 - AI工具集合
    ENAIAreaModeHistory,      // 历史模式 - 交互历史记录
    ENAIAreaModeSettings      // 设置模式 - AI功能设置
};

@interface ENAIAreaViewController : NSViewController

// 核心属性
@property (strong, readonly) ENAccountController *accountController;
@property (strong, nonatomic) ENMacNoteMO *currentNote;
@property (assign, nonatomic) ENAIAreaMode currentMode;

// 子控制器
@property (strong, readonly) ENAIHeaderViewController *headerViewController;
@property (strong, readonly) ENAIChatViewController *chatViewController;
@property (strong, readonly) ENAIToolsViewController *toolsViewController;
@property (strong, readonly) ENAIHistoryViewController *historyViewController;
@property (strong, readonly) ENAISettingsViewController *settingsViewController;

// XIB 连接
@property (weak) IBOutlet NSView *aiHeaderView;
@property (weak) IBOutlet NSView *aiContentView;
@property (weak) IBOutlet NSView *aiFooterView;

// 委托
@property (weak) id<ENAIAreaDelegate> delegate;

// 初始化
- (instancetype)initWithAccountController:(ENAccountController *)accountController;

// 控制方法
- (void)setCurrentNote:(ENMacNoteMO *)note;
- (void)switchToMode:(ENAIAreaMode)mode animated:(BOOL)animated;
- (void)refreshContent;

// 生命周期
- (void)cleanup;

@end

// AI 区域委托协议
@protocol ENAIAreaDelegate <NSObject>
@optional
- (void)aiAreaDidRequestMinimize:(ENAIAreaViewController *)aiArea;
- (void)aiAreaDidChangeMode:(ENAIAreaViewController *)aiArea toMode:(ENAIAreaMode)mode;
- (void)aiArea:(ENAIAreaViewController *)aiArea didRequestInsertText:(NSString *)text;
- (void)aiArea:(ENAIAreaViewController *)aiArea didRequestReplaceSelection:(NSString *)text;
- (NSString *)aiAreaRequestedSelectedText:(ENAIAreaViewController *)aiArea;
- (NSString *)aiAreaRequestedNoteContent:(ENAIAreaViewController *)aiArea;
@end
```

#### 3.2.2 模式切换实现

```objc
@implementation ENAIAreaViewController

- (void)switchToMode:(ENAIAreaMode)mode animated:(BOOL)animated {
    if (self.currentMode == mode) {
        return;
    }
    
    ENAIAreaMode previousMode = self.currentMode;
    self.currentMode = mode;
    
    // 移除当前显示的视图控制器
    [self hideCurrentViewController:animated];
    
    // 显示新的视图控制器
    [self showViewControllerForMode:mode animated:animated];
    
    // 更新头部选择状态
    [self.headerViewController setSelectedMode:mode];
    
    // 保存模式到偏好设置
    [[NSUserDefaults standardUserDefaults] setInteger:mode forKey:ENAIAreaLastModeKey];
    
    // 通知委托
    if ([self.delegate respondsToSelector:@selector(aiAreaDidChangeMode:toMode:)]) {
        [self.delegate aiAreaDidChangeMode:self toMode:mode];
    }
}

- (void)showViewControllerForMode:(ENAIAreaMode)mode animated:(BOOL)animated {
    NSViewController *targetViewController = [self viewControllerForMode:mode];
    
    if (!targetViewController) {
        return;
    }
    
    // 添加子视图控制器
    [self addChildViewController:targetViewController];
    [self.aiContentView addSubview:targetViewController.view];
    
    // 设置约束
    targetViewController.view.translatesAutoresizingMaskIntoConstraints = NO;
    [NSLayoutConstraint activateConstraints:@[
        [targetViewController.view.topAnchor constraintEqualToAnchor:self.aiContentView.topAnchor],
        [targetViewController.view.leadingAnchor constraintEqualToAnchor:self.aiContentView.leadingAnchor],
        [targetViewController.view.trailingAnchor constraintEqualToAnchor:self.aiContentView.trailingAnchor],
        [targetViewController.view.bottomAnchor constraintEqualToAnchor:self.aiContentView.bottomAnchor]
    ]];
    
    if (animated) {
        // 淡入动画
        targetViewController.view.alphaValue = 0.0;
        [NSAnimationContext runAnimationGroup:^(NSAnimationContext *context) {
            context.duration = 0.25;
            targetViewController.view.animator.alphaValue = 1.0;
        } completionHandler:nil];
    }
}

- (NSViewController *)viewControllerForMode:(ENAIAreaMode)mode {
    switch (mode) {
        case ENAIAreaModeChat:
            return self.chatViewController;
        case ENAIAreaModeTools:
            return self.toolsViewController;
        case ENAIAreaModeHistory:
            return self.historyViewController;
        case ENAIAreaModeSettings:
            return self.settingsViewController;
        default:
            return nil;
    }
}

@end
```

### 3.3 ENAIHeaderViewController - AI 头部控制器

#### 3.3.1 类定义

```objc
@interface ENAIHeaderViewController : NSViewController

// XIB 连接
@property (weak) IBOutlet NSSegmentedControl *modeSelector;
@property (weak) IBOutlet NSButton *minimizeButton;
@property (weak) IBOutlet NSButton *settingsButton;
@property (weak) IBOutlet NSTextField *titleLabel;
@property (weak) IBOutlet NSView *connectionStatusView;

// 委托
@property (weak) id<ENAIHeaderDelegate> delegate;

// 状态
@property (assign) ENAIAreaMode selectedMode;
@property (assign) BOOL isConnected;

// 动作
- (IBAction)modeSelectorChanged:(NSSegmentedControl *)sender;
- (IBAction)minimizeButtonClicked:(NSButton *)sender;
- (IBAction)settingsButtonClicked:(NSButton *)sender;

// 控制方法
- (void)setSelectedMode:(ENAIAreaMode)mode;
- (void)setConnectionStatus:(BOOL)connected;
- (void)updateTitle:(NSString *)title;

@end

@protocol ENAIHeaderDelegate <NSObject>
- (void)aiHeaderDidSelectMode:(ENAIAreaMode)mode;
- (void)aiHeaderDidRequestMinimize;
- (void)aiHeaderDidRequestSettings;
@end
```

#### 3.3.2 头部布局设计

```
ENAIHeaderViewController.view [280x44]
├── titleLabel [80x20] - "AI Assistant"
├── connectionStatusView [16x16] - 连接状态指示器
├── modeSelector [120x24] - [Chat|Tools|History]分段控制
├── settingsButton [24x24] - ⚙️ 设置按钮
└── minimizeButton [24x24] - ➖ 最小化按钮
```

### 3.4 ENAIChatViewController - AI 聊天控制器

#### 3.4.1 类定义

```objc
@interface ENAIChatViewController : NSViewController <NSTableViewDataSource, NSTableViewDelegate, NSTextFieldDelegate>

// XIB 连接
@property (weak) IBOutlet NSScrollView *chatScrollView;
@property (weak) IBOutlet NSTableView *chatTableView;
@property (weak) IBOutlet NSView *inputContainerView;
@property (weak) IBOutlet NSTextField *inputTextField;
@property (weak) IBOutlet NSButton *sendButton;
@property (weak) IBOutlet NSProgressIndicator *thinkingIndicator;
@property (weak) IBOutlet NSButton *clearHistoryButton;

// 数据和服务
@property (strong) NSMutableArray<ENAIChatMessage *> *chatMessages;
@property (strong) ENAIService *aiService;

// 状态
@property (assign) BOOL isWaitingForResponse;
@property (strong) ENMacNoteMO *currentNote;

// 委托
@property (weak) id<ENAIChatDelegate> delegate;

// 方法
- (void)sendMessage:(NSString *)message;
- (void)receiveAIResponse:(NSString *)response;
- (void)showThinkingIndicator:(BOOL)show;
- (void)clearChatHistory;
- (void)setCurrentNote:(ENMacNoteMO *)note;

// 动作
- (IBAction)sendButtonClicked:(NSButton *)sender;
- (IBAction)clearHistoryButtonClicked:(NSButton *)sender;

@end

// 聊天委托协议
@protocol ENAIChatDelegate <NSObject>
@optional
- (void)aiChatDidRequestInsertText:(NSString *)text;
- (void)aiChatDidRequestReplaceSelection:(NSString *)text;
- (NSString *)aiChatRequestedSelectedText;
- (NSString *)aiChatRequestedNoteContent;
@end
```

#### 3.4.2 聊天消息数据模型

```objc
@interface ENAIChatMessage : NSObject

@property (strong, nonatomic) NSString *messageId;
@property (strong, nonatomic) NSString *content;
@property (strong, nonatomic) NSDate *timestamp;
@property (assign, nonatomic) BOOL isFromAI;
@property (assign, nonatomic) ENAIChatMessageType messageType;
@property (strong, nonatomic, nullable) NSDictionary *metadata;

- (instancetype)initWithContent:(NSString *)content isFromAI:(BOOL)isFromAI;
- (instancetype)initWithContent:(NSString *)content 
                        isFromAI:(BOOL)isFromAI 
                     messageType:(ENAIChatMessageType)messageType;

@end

typedef NS_ENUM(NSUInteger, ENAIChatMessageType) {
    ENAIChatMessageTypeText = 0,      // 普通文本消息
    ENAIChatMessageTypeCode,          // 代码片段
    ENAIChatMessageTypeMarkdown,      // Markdown 内容
    ENAIChatMessageTypeAction,        // 操作建议
    ENAIChatMessageTypeError          // 错误消息
};
```

#### 3.4.3 聊天界面布局

```
ENAIChatViewController.view [280x700]
├── chatScrollView [280x620]
│   └── chatTableView
│       ├── ENAIChatMessageCellView (用户消息)
│       │   ├── avatarImageView [32x32]
│       │   ├── bubbleView [背景气泡]
│       │   │   └── messageLabel [消息文本]
│       │   └── timestampLabel [时间戳]
│       ├── ENAIChatMessageCellView (AI消息)
│       │   ├── avatarImageView [32x32] - AI头像
│       │   ├── bubbleView [背景气泡]
│       │   │   ├── messageLabel [消息文本]
│       │   │   └── actionButtons [操作按钮]
│       │   └── timestampLabel [时间戳]
│       └── ... (更多消息)
├── inputContainerView [280x80]
│   ├── inputTextField [200x32] - 消息输入框
│   ├── sendButton [32x32] - 发送按钮
│   ├── thinkingIndicator [24x24] - 思考指示器
│   └── clearHistoryButton [24x24] - 清空历史按钮
```

### 3.5 ENAIToolsViewController - AI 工具控制器

#### 3.5.1 类定义

```objc
@interface ENAIToolsViewController : NSViewController <NSCollectionViewDataSource, NSCollectionViewDelegate>

// XIB 连接
@property (weak) IBOutlet NSCollectionView *toolsCollectionView;
@property (weak) IBOutlet NSTextField *categoryLabel;
@property (weak) IBOutlet NSSegmentedControl *categorySelector;

// 数据
@property (strong) NSArray<ENAITool *> *availableTools;
@property (strong) NSArray<ENAIToolCategory *> *toolCategories;
@property (assign) ENAIToolCategory selectedCategory;

// 状态
@property (strong) ENMacNoteMO *currentNote;

// 委托
@property (weak) id<ENAIToolsDelegate> delegate;

// 方法
- (void)refreshTools;
- (void)setCurrentNote:(ENMacNoteMO *)note;
- (void)executeTool:(ENAITool *)tool withParameters:(NSDictionary *)parameters;

// 动作
- (IBAction)categorySelectorChanged:(NSSegmentedControl *)sender;

@end

@protocol ENAIToolsDelegate <NSObject>
@optional
- (void)aiToolsDidExecuteTool:(ENAITool *)tool withResult:(id)result;
- (void)aiToolsDidRequestNoteContent;
- (void)aiToolsDidRequestInsertContent:(NSString *)content;
@end
```

#### 3.5.2 AI 工具数据模型

```objc
@interface ENAITool : NSObject

@property (strong, nonatomic) NSString *toolId;
@property (strong, nonatomic) NSString *name;
@property (strong, nonatomic) NSString *toolDescription;
@property (strong, nonatomic) NSImage *icon;
@property (assign, nonatomic) ENAIToolCategory category;
@property (strong, nonatomic) NSArray<ENAIToolParameter *> *parameters;
@property (assign, nonatomic) BOOL requiresNoteContent;
@property (assign, nonatomic) BOOL requiresSelection;

- (BOOL)canExecuteWithCurrentState;
- (void)executeWithParameters:(NSDictionary *)params 
                   completion:(void(^)(id result, NSError *error))completion;

@end

typedef NS_ENUM(NSUInteger, ENAIToolCategory) {
    ENAIToolCategoryWriting = 0,      // 写作辅助工具
    ENAIToolCategoryAnalysis,         // 分析工具  
    ENAIToolCategoryTranslation,      // 翻译工具
    ENAIToolCategoryFormatting,       // 格式化工具
    ENAIToolCategorySummarization,    // 摘要工具
    ENAIToolCategoryCustom            // 自定义工具
};

// 预定义的 AI 工具
extern NSString * const ENAIToolSummarizeId;         // "summarize"
extern NSString * const ENAIToolTranslateId;         // "translate" 
extern NSString * const ENAIToolGrammarCheckId;      // "grammar_check"
extern NSString * const ENAIToolExpandContentId;     // "expand_content"
extern NSString * const ENAIToolExtractKeywordsId;   // "extract_keywords"
extern NSString * const ENAIToolGenerateOutlineId;   // "generate_outline"
```

#### 3.5.3 工具界面布局

```
ENAIToolsViewController.view [280x700]
├── categorySelector [200x24] - 工具分类选择
├── categoryLabel [280x20] - 当前分类标题
└── toolsCollectionView [280x656] - 工具网格
    ├── ENAIToolCollectionItem [130x100] - 摘要工具
    │   ├── iconImageView [48x48]
    │   ├── nameLabel [120x16]
    │   └── descriptionLabel [120x32]
    ├── ENAIToolCollectionItem [130x100] - 翻译工具
    ├── ENAIToolCollectionItem [130x100] - 语法检查
    ├── ENAIToolCollectionItem [130x100] - 内容扩展
    └── ... (更多工具)
```

## 四、EN4NotesModule 的核心改动

### 4.1 扩展属性和方法

```objc
// EN4NotesModule.h 中的新增内容
@interface EN4NotesModule : ENBaseViewModule

// 现有属性保持不变...
@property (weak) IBOutlet ENSplitView *splitView;
@property (weak) IBOutlet NSView *contentContainerView;
@property (strong) ENNoteListContainerViewController *noteListContainerViewController;

// 新增 AI 相关属性
@property (strong) ENEditorAIContainerViewController *editorAIContainerViewController;
@property (assign) BOOL isAIAreaVisible;
@property (assign) CGFloat aiAreaWidth;
@property (strong, readonly) ENAIAreaViewController *aiAreaViewController;

// 新增方法
- (void)showAIArea:(BOOL)show;
- (void)showAIArea:(BOOL)show animated:(BOOL)animated;
- (void)toggleAIArea;
- (void)setAIAreaWidth:(CGFloat)width;
- (void)setAIAreaWidth:(CGFloat)width animated:(BOOL)animated;

// AI 功能集成
- (void)aiDidRequestInsertText:(NSString *)text;
- (void)aiDidRequestReplaceSelection:(NSString *)text;
- (NSString *)getSelectedTextForAI;
- (NSString *)getNoteContentForAI;

@end
```

### 4.2 AI Area 显示/隐藏控制实现

```objc
@implementation EN4NotesModule

- (void)showAIArea:(BOOL)show animated:(BOOL)animated {
    if (self.isAIAreaVisible == show) {
        return;
    }
    
    self.isAIAreaVisible = show;
    
    if (show) {
        [self createEditorAIContainerIfNeeded];
        [self replaceEditorWithContainerAnimated:animated];
        [self restoreAIAreaWidth];
    } else {
        [self restoreOriginalEditorLayoutAnimated:animated];
    }
    
    // 保存状态
    [[NSUserDefaults standardUserDefaults] setBool:show forKey:ENAIAreaVisibleKey];
    
    // 发送通知
    [[NSNotificationCenter defaultCenter] 
        postNotificationName:ENAIAreaVisibilityDidChangeNotification 
                      object:self 
                    userInfo:@{@"visible": @(show)}];
}

- (void)createEditorAIContainerIfNeeded {
    if (!self.editorAIContainerViewController) {
        // 创建容器控制器
        self.editorAIContainerViewController = 
            [[ENEditorAIContainerViewController alloc] 
                initWithEditorViewController:self.editorView 
                           accountController:self.accountController];
        
        // 设置委托
        self.editorAIContainerViewController.delegate = self;
        
        // 配置 AI 区域
        ENAIAreaViewController *aiArea = self.editorAIContainerViewController.aiAreaViewController;
        aiArea.delegate = self;
        [aiArea setCurrentNote:self.currentNote];
    }
}

- (void)replaceEditorWithContainerAnimated:(BOOL)animated {
    // 保存当前编辑器的父视图
    NSView *originalParent = self.editorView.view.superview;
    NSRect originalFrame = self.editorView.view.frame;
    
    // 移除原编辑器视图
    [self.editorView.view removeFromSuperview];
    
    // 添加容器视图控制器
    [self addChildViewController:self.editorAIContainerViewController];
    [originalParent addSubview:self.editorAIContainerViewController.view];
    
    // 设置约束
    self.editorAIContainerViewController.view.frame = originalFrame;
    self.editorAIContainerViewController.view.autoresizingMask = 
        NSViewWidthSizable | NSViewHeightSizable;
    
    if (animated) {
        // AI 区域从右侧滑入的动画
        [self animateAIAreaSlideIn];
    }
}

- (void)animateAIAreaSlideIn {
    NSView *aiAreaView = self.editorAIContainerViewController.aiAreaContainerView;
    
    // 初始状态：AI 区域在右侧屏幕外
    NSRect originalFrame = aiAreaView.frame;
    NSRect startFrame = originalFrame;
    startFrame.origin.x = originalFrame.origin.x + originalFrame.size.width;
    aiAreaView.frame = startFrame;
    
    // 动画到最终位置
    [NSAnimationContext runAnimationGroup:^(NSAnimationContext *context) {
        context.duration = 0.3;
        context.timingFunction = [CAMediaTimingFunction functionWithName:kCAMediaTimingFunctionEaseOut];
        aiAreaView.animator.frame = originalFrame;
    } completionHandler:nil];
}

@end
```

### 4.3 NotesView4.xib 的修改

```xml
<!-- 修改 NotesView4.xib 中的主要分割视图结构 -->
<splitView dividerStyle="thin" vertical="YES" translatesAutoresizingMaskIntoConstraints="NO" id="10" customClass="ENSplitView">
    <subviews>
        <!-- 左侧：笔记列表区域 (保持原有结构不变) -->
        <stackView distribution="fill" orientation="vertical" alignment="leading" spacing="0.0" 
                   horizontalStackHuggingPriority="249.99998474121094" 
                   verticalStackHuggingPriority="249.99998474121094" 
                   fixedFrame="YES" detachesHiddenViews="YES" id="7Zc-x4-9MD">
            <!-- 现有的笔记列表组件保持不变... -->
        </stackView>
        
        <!-- 右侧：内容容器 (从固定的编辑器改为动态容器) -->
        <customView identifier="contentContainer" fixedFrame="YES" id="12" userLabel="Content Container">
            <rect key="frame" x="293" y="0.0" width="434" height="798"/>
            <autoresizingMask key="autoresizingMask"/>
            <!-- 这里将动态加载编辑器或编辑器+AI 容器 -->
        </customView>
    </subviews>
    <holdingPriorities>
        <real value="275"/>  <!-- 笔记列表优先级 -->
        <real value="250"/>  <!-- 内容区域优先级 -->
    </holdingPriorities>
    <connections>
        <outlet property="delegate" destination="-2" id="357"/>
    </connections>
</splitView>
```

## 五、工具栏集成

### 5.1 SuperNoteEditorToolbarViewController 扩展

```swift
// SuperNoteEditorToolbarViewController.swift 中的新增内容

// 新增 AI 控制按钮
@IBOutlet weak var aiToggleButton: NSButton!
@IBOutlet weak var aiModeSelector: NSSegmentedControl!

// 新增委托方法
extension SuperNoteToolbarDelegate {
    @objc optional func toggleAIArea(_ sender: Any)
    @objc optional func setAIMode(_ mode: Int, sender: Any)
}

// 实现 AI 控制动作
@IBAction func toggleAIArea(_ sender: NSButton) {
    delegate?.toggleAIArea?(sender)
    
    // 更新按钮状态
    updateAIToggleButtonState()
}

@IBAction func aiModeSelectorChanged(_ sender: NSSegmentedControl) {
    delegate?.setAIMode?(sender.selectedSegment, sender: sender)
}

// 更新 AI 按钮状态
func updateAIToggleButtonState() {
    guard let notesModule = delegate as? EN4NotesModule else { return }
    
    aiToggleButton.state = notesModule.isAIAreaVisible ? .on : .off
    aiModeSelector.isHidden = !notesModule.isAIAreaVisible
    
    // 更新工具栏布局
    configureToolbar()
}
```

### 5.2 工具栏 XIB 文件修改

在 `SuperNoteEditorToolbarViewController.xib` 中添加 AI 相关控件：

```xml
<!-- AI 切换按钮 -->
<button verticalHuggingPriority="750" translatesAutoresizingMaskIntoConstraints="NO" 
        id="aiToggleButton" userLabel="AI Toggle Button">
    <rect key="frame" x="320" y="12" width="32" height="20"/>
    <buttonCell key="cell" type="toggle" title="AI" bezelStyle="rounded" 
                alignment="center" controlSize="small" state="on" 
                borderStyle="border" imageScaling="proportionallyDown" inset="2" 
                id="aiToggleButtonCell">
        <behavior key="behavior" pushIn="YES" lightByBackground="YES" lightByGray="YES"/>
        <font key="font" metaFont="system"/>
    </buttonCell>
    <connections>
        <action selector="toggleAIArea:" target="-2" id="aiToggleAction"/>
    </connections>
</button>

<!-- AI 模式选择器 -->
<segmentedControl verticalHuggingPriority="750" translatesAutoresizingMaskIntoConstraints="NO" 
                  id="aiModeSelector" userLabel="AI Mode Selector" hidden="YES">
    <rect key="frame" x="356" y="12" width="120" height="20"/>
    <segmentedCell key="cell" controlSize="small" borderStyle="border" 
                   alignment="left" style="roundRect" trackingMode="selectOne" 
                   id="aiModeSelectorCell">
        <font key="font" metaFont="system"/>
        <segments>
            <segment label="Chat" width="40" selected="YES"/>
            <segment label="Tools" width="40" tag="1"/>
            <segment label="History" width="40" tag="2"/>
        </segments>
    </segmentedCell>
    <connections>
        <action selector="aiModeSelectorChanged:" target="-2" id="aiModeAction"/>
    </connections>
</segmentedControl>
```

## 六、快捷键和菜单集成

### 6.1 快捷键支持

```objc
// ENNoteCollectionWindowController 中添加 AI 相关快捷键
- (void)keyDown:(NSEvent *)event {
    NSString *characters = event.charactersIgnoringModifiers;
    NSEventModifierFlags flags = event.modifierFlags;
    
    // Cmd+Shift+A: 切换 AI 区域
    if ((flags & NSEventModifierFlagCommand) && 
        (flags & NSEventModifierFlagShift) &&
        [characters isEqualToString:@"a"]) {
        [self toggleAIArea:nil];
        return;
    }
    
    // Cmd+1/2/3: 快速切换 AI 模式
    if ((flags & NSEventModifierFlagCommand) && self.isAIAreaVisible) {
        if ([characters isEqualToString:@"1"]) {
            [self setAIMode:ENAIAreaModeChat];
            return;
        } else if ([characters isEqualToString:@"2"]) {
            [self setAIMode:ENAIAreaModeTools];
            return;
        } else if ([characters isEqualToString:@"3"]) {
            [self setAIMode:ENAIAreaModeHistory];
            return;
        }
    }
    
    [super keyDown:event];
}

// AI 控制方法
- (IBAction)toggleAIArea:(id)sender {
    EN4NotesModule *notesModule = (EN4NotesModule *)self.viewModulesManager.currentModule;
    if ([notesModule isKindOfClass:[EN4NotesModule class]]) {
        [notesModule toggleAIArea];
    }
}

- (void)setAIMode:(ENAIAreaMode)mode {
    EN4NotesModule *notesModule = (EN4NotesModule *)self.viewModulesManager.currentModule;
    if ([notesModule isKindOfClass:[EN4NotesModule class]] && notesModule.isAIAreaVisible) {
        [notesModule.aiAreaViewController switchToMode:mode animated:YES];
    }
}
```

### 6.2 菜单项集成

在主菜单中添加 AI 相关菜单项：

```xml
<!-- 在 View 菜单中添加 AI 相关项 -->
<menu title="View">
    <!-- 现有菜单项... -->
    <menuItem isSeparatorItem="YES"/>
    <menuItem title="Show AI Assistant" keyEquivalent="a" 
              keyEquivalentModifierMask="1179648" id="showAIMenuItem">
        <connections>
            <action selector="toggleAIArea:" target="-1" id="showAIAction"/>
        </connections>
    </menuItem>
    <menu title="AI Mode" id="aiModeSubmenu">
        <menuItem title="Chat" keyEquivalent="1" 
                  keyEquivalentModifierMask="1048576" tag="0" id="aiChatModeMenuItem">
            <connections>
                <action selector="setAIModeFromMenu:" target="-1" id="aiChatModeAction"/>
            </connections>
        </menuItem>
        <menuItem title="Tools" keyEquivalent="2" 
                  keyEquivalentModifierMask="1048576" tag="1" id="aiToolsModeMenuItem">
            <connections>
                <action selector="setAIModeFromMenu:" target="-1" id="aiToolsModeAction"/>
            </connections>
        </menuItem>
        <menuItem title="History" keyEquivalent="3" 
                  keyEquivalentModifierMask="1048576" tag="2" id="aiHistoryModeMenuItem">
            <connections>
                <action selector="setAIModeFromMenu:" target="-1" id="aiHistoryModeAction"/>
            </connections>
        </menuItem>
    </menu>
</menu>
```

## 七、AI 服务集成

### 7.1 ENAIService 核心服务

```objc
@interface ENAIService : NSObject

// 单例
+ (instancetype)sharedService;

// 核心属性
@property (strong, readonly) ENAccountController *accountController;
@property (assign, readonly) BOOL isConnected;
@property (assign, readonly) BOOL isProcessing;

// 聊天功能
- (void)sendChatMessage:(NSString *)message 
            withContext:(ENAIChatContext *)context
             completion:(void(^)(NSString *response, NSError *error))completion;

// 工具执行
- (void)executeTool:(ENAITool *)tool 
     withParameters:(NSDictionary *)parameters
            context:(ENAIContext *)context
         completion:(void(^)(id result, NSError *error))completion;

// 内容分析
- (void)analyzeContent:(NSString *)content 
            completion:(void(^)(ENAIAnalysisResult *result, NSError *error))completion;

// 连接管理
- (void)connect;
- (void)disconnect;

@end

// AI 上下文信息
@interface ENAIContext : NSObject
@property (strong) ENMacNoteMO *currentNote;
@property (strong) NSString *selectedText;
@property (strong) NSString *noteContent;
@property (strong) NSDictionary *userPreferences;
@end
```

### 7.2 AI 功能预设

```objc
// 预定义的 AI 工具配置
@implementation ENAIToolsManager

+ (NSArray<ENAITool *> *)defaultTools {
    return @[
        [self createSummarizeTool],
        [self createTranslateTool],
        [self createGrammarCheckTool],
        [self createExpandContentTool],
        [self createExtractKeywordsTool],
        [self createGenerateOutlineTool]
    ];
}

+ (ENAITool *)createSummarizeTool {
    ENAITool *tool = [[ENAITool alloc] init];
    tool.toolId = ENAIToolSummarizeId;
    tool.name = @"Summarize";
    tool.toolDescription = @"Generate a concise summary of the selected content";
    tool.icon = [NSImage imageNamed:@"ai_summarize_icon"];
    tool.category = ENAIToolCategorySummarization;
    tool.requiresNoteContent = YES;
    tool.requiresSelection = NO;
    return tool;
}

+ (ENAITool *)createTranslateTool {
    ENAITool *tool = [[ENAITool alloc] init];
    tool.toolId = ENAIToolTranslateId;
    tool.name = @"Translate";
    tool.toolDescription = @"Translate text to different languages";
    tool.icon = [NSImage imageNamed:@"ai_translate_icon"];
    tool.category = ENAIToolCategoryTranslation;
    tool.requiresSelection = YES;
    
    // 添加参数
    ENAIToolParameter *targetLang = [[ENAIToolParameter alloc] init];
    targetLang.name = @"target_language";
    targetLang.displayName = @"Target Language";
    targetLang.type = ENAIToolParameterTypeChoice;
    targetLang.choices = @[@"English", @"Chinese", @"Spanish", @"French", @"German"];
    tool.parameters = @[targetLang];
    
    return tool;
}

@end
```

## 八、性能优化和内存管理

### 8.1 懒加载策略

```objc
@implementation EN4NotesModule

// AI 区域控制器的懒加载
- (ENAIAreaViewController *)aiAreaViewController {
    if (!self.editorAIContainerViewController) {
        return nil;
    }
    return self.editorAIContainerViewController.aiAreaViewController;
}

// 编辑器+AI容器的懒加载
- (ENEditorAIContainerViewController *)editorAIContainerViewController {
    if (!_editorAIContainerViewController && self.isAIAreaVisible) {
        _editorAIContainerViewController = 
            [[ENEditorAIContainerViewController alloc] 
                initWithEditorViewController:self.editorView 
                           accountController:self.accountController];
    }
    return _editorAIContainerViewController;
}

@end

@implementation ENAIAreaViewController

// 子控制器的懒加载
- (ENAIChatViewController *)chatViewController {
    if (!_chatViewController) {
        _chatViewController = [[ENAIChatViewController alloc] init];
        _chatViewController.delegate = self;
    }
    return _chatViewController;
}

- (ENAIToolsViewController *)toolsViewController {
    if (!_toolsViewController) {
        _toolsViewController = [[ENAIToolsViewController alloc] init];
        _toolsViewController.delegate = self;
    }
    return _toolsViewController;
}

@end
```

### 8.2 内存管理策略

```objc
@implementation ENAIAreaViewController

- (void)cleanup {
    // 清理聊天历史
    [self.chatViewController clearChatHistory];
    
    // 停止所有正在进行的 AI 请求
    [[ENAIService sharedService] cancelAllRequests];
    
    // 释放子控制器
    [self.chatViewController removeFromParentViewController];
    [self.toolsViewController removeFromParentViewController];
    [self.historyViewController removeFromParentViewController];
    
    _chatViewController = nil;
    _toolsViewController = nil;
    _historyViewController = nil;
}

- (void)viewDidDisappear {
    [super viewDidDisappear];
    
    // 如果 AI 区域被隐藏，进行内存清理
    if (!self.parentViewController) {
        [self cleanup];
    }
}

@end

@implementation EN4NotesModule

- (void)moduleWillDisappear {
    [super moduleWillDisappear];
    
    // 模块切换时保存 AI 区域状态并清理资源
    if (self.isAIAreaVisible) {
        [self saveAIAreaState];
    }
    
    if (self.editorAIContainerViewController) {
        [self.editorAIContainerViewController.aiAreaViewController cleanup];
    }
}

@end
```

### 8.3 响应式设计

```objc
@implementation ENEditorAIContainerViewController

- (void)viewDidLayout {
    [super viewDidLayout];
    
    // 根据窗口大小自动调整布局
    [self adjustLayoutForWindowSize];
}

- (void)adjustLayoutForWindowSize {
    CGFloat totalWidth = self.view.frame.size.width;
    
    // 如果窗口太小，自动隐藏 AI 区域
    if (totalWidth < EN_MINIMUM_WINDOW_WIDTH_FOR_AI) {
        if (self.delegate && [self.delegate respondsToSelector:@selector(shouldAutoHideAIArea)]) {
            [self.delegate performSelector:@selector(shouldAutoHideAIArea)];
        }
        return;
    }
    
    // 调整最小宽度约束
    CGFloat editorMinWidth = MAX(EN_EDITOR_MIN_WIDTH, totalWidth * 0.5);
    CGFloat aiMaxWidth = MIN(EN_AI_AREA_MAX_WIDTH, totalWidth * 0.4);
    
    [self updateSplitViewConstraintsWithEditorMinWidth:editorMinWidth aiMaxWidth:aiMaxWidth];
}

@end
```

## 九、状态持久化

### 9.1 用户偏好设置

```objc
// ENAIPreferences.h
@interface ENAIPreferences : NSObject

// 单例
+ (instancetype)sharedPreferences;

// AI 区域设置
@property (assign) BOOL isAIAreaVisible;
@property (assign) CGFloat aiAreaWidth;
@property (assign) ENAIAreaMode lastSelectedMode;

// AI 功能设置
@property (assign) BOOL enableAutoSuggestions;
@property (assign) BOOL enableContextualTools;
@property (strong) NSString *preferredLanguage;
@property (assign) ENAIResponseStyle responseStyle;

// 保存和恢复
- (void)save;
- (void)restore;

@end

// 偏好设置键常量
extern NSString * const ENAIAreaVisibleKey;        // "ENAIAreaVisible"
extern NSString * const ENAIAreaWidthKey;          // "ENAIAreaWidth"
extern NSString * const ENAIAreaLastModeKey;       // "ENAIAreaLastMode"
extern NSString * const ENAIAutoSuggestionsKey;    // "ENAIAutoSuggestions"
extern NSString * const ENAIContextualToolsKey;    // "ENAIContextualTools"
extern NSString * const ENAIPreferredLanguageKey;  // "ENAIPreferredLanguage"

#define EN_MINIMUM_WINDOW_WIDTH_FOR_AI  800.0f
```

### 9.2 状态恢复实现

```objc
@implementation EN4NotesModule

- (void)awakeFromNib {
    [super awakeFromNib];
    
    // 恢复 AI 区域状态
    [self restoreAIAreaState];
}

- (void)restoreAIAreaState {
    ENAIPreferences *prefs = [ENAIPreferences sharedPreferences];
    
    // 恢复可见性
    BOOL shouldShow = prefs.isAIAreaVisible;
    if (shouldShow) {
        [self showAIArea:YES animated:NO];
        
        // 恢复宽度
        if (prefs.aiAreaWidth > 0) {
            [self setAIAreaWidth:prefs.aiAreaWidth animated:NO];
        }
        
        // 恢复模式
        [self.aiAreaViewController switchToMode:prefs.lastSelectedMode animated:NO];
    }
}

- (void)saveAIAreaState {
    ENAIPreferences *prefs = [ENAIPreferences sharedPreferences];
    
    prefs.isAIAreaVisible = self.isAIAreaVisible;
    prefs.aiAreaWidth = self.aiAreaWidth;
    
    if (self.aiAreaViewController) {
        prefs.lastSelectedMode = self.aiAreaViewController.currentMode;
    }
    
    [prefs save];
}

@end
```

## 十、通知和事件系统

### 10.1 通知常量定义

```objc
// ENAINotifications.h
extern NSString * const ENAIAreaVisibilityDidChangeNotification;     // AI区域显示状态变化
extern NSString * const ENAIAreaModeDidChangeNotification;           // AI区域模式变化
extern NSString * const ENAIAreaLayoutDidChangeNotification;         // AI区域布局变化
extern NSString * const ENAIChatMessageDidSendNotification;          // AI聊天消息发送
extern NSString * const ENAIChatMessageDidReceiveNotification;       // AI聊天消息接收
extern NSString * const ENAIToolDidExecuteNotification;              // AI工具执行
extern NSString * const ENAIServiceConnectionDidChangeNotification;  // AI服务连接状态变化
```

### 10.2 事件处理实现

```objc
@implementation EN4NotesModule

- (void)viewDidLoad {
    [super viewDidLoad];
    
    // 注册通知监听
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(aiServiceConnectionDidChange:)
                                                 name:ENAIServiceConnectionDidChangeNotification
                                               object:nil];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(aiAreaLayoutDidChange:)
                                                 name:ENAIAreaLayoutDidChangeNotification
                                               object:nil];
}

- (void)dealloc {
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

- (void)aiServiceConnectionDidChange:(NSNotification *)notification {
    BOOL isConnected = [notification.userInfo[@"connected"] boolValue];
    
    // 更新 AI 区域的连接状态显示
    if (self.aiAreaViewController) {
        [self.aiAreaViewController.headerViewController setConnectionStatus:isConnected];
    }
    
    // 如果连接断开，显示重连提示
    if (!isConnected && self.isAIAreaVisible) {
        [self showAIConnectionLostAlert];
    }
}

- (void)aiAreaLayoutDidChange:(NSNotification *)notification {
    // 保存布局变化
    [self saveAIAreaState];
    
    // 通知编辑器调整布局
    [self.editorView updateNoteEditorBackground];
}

@end
```

## 十一、测试和验证

### 11.1 单元测试覆盖

```objc
// ENAIAreaTests.m
@interface ENAIAreaTests : XCTestCase
@property (strong) ENAIAreaViewController *aiAreaViewController;
@property (strong) ENAccountController *mockAccountController;
@end

@implementation ENAIAreaTests

- (void)setUp {
    [super setUp];
    
    self.mockAccountController = [[ENAccountController alloc] init];
    self.aiAreaViewController = [[ENAIAreaViewController alloc] 
                                initWithAccountController:self.mockAccountController];
}

- (void)testAIAreaInitialization {
    XCTAssertNotNil(self.aiAreaViewController);
    XCTAssertEqual(self.aiAreaViewController.currentMode, ENAIAreaModeChat);
    XCTAssertNotNil(self.aiAreaViewController.headerViewController);
}

- (void)testModeSwitch {
    [self.aiAreaViewController switchToMode:ENAIAreaModeTools animated:NO];
    XCTAssertEqual(self.aiAreaViewController.currentMode, ENAIAreaModeTools);
}

- (void)testAIAreaVisibilityToggle {
    EN4NotesModule *notesModule = [[EN4NotesModule alloc] init];
    
    // 初始状态应该是隐藏的
    XCTAssertFalse(notesModule.isAIAreaVisible);
    
    // 显示 AI 区域
    [notesModule showAIArea:YES];
    XCTAssertTrue(notesModule.isAIAreaVisible);
    
    // 隐藏 AI 区域
    [notesModule showAIArea:NO];
    XCTAssertFalse(notesModule.isAIAreaVisible);
}

@end
```

### 11.2 集成测试场景

1. **布局调整测试**: 验证拖拽调整 AI 区域大小的功能
2. **模式切换测试**: 验证在不同 AI 模式间切换的流畅性
3. **状态持久化测试**: 验证 AI 区域设置的保存和恢复
4. **性能测试**: 验证 AI 区域显示/隐藏的性能影响
5. **内存泄漏测试**: 验证 AI 组件的内存管理

## 十二、部署和发布

### 12.1 渐进式发布策略

1. **阶段一**: 基础 AI 区域和聊天功能
2. **阶段二**: AI 工具集成和历史记录
3. **阶段三**: 高级 AI 功能和自定义工具
4. **阶段四**: 性能优化和用户体验改进

### 12.2 功能开关配置

```objc
// ENFeatureFlags.h
@interface ENFeatureFlags : NSObject

+ (BOOL)isAIAreaEnabled;                    // AI 区域总开关
+ (BOOL)isAIChatEnabled;                    // AI 聊天功能
+ (BOOL)isAIToolsEnabled;                   // AI 工具功能
+ (BOOL)isAIHistoryEnabled;                 // AI 历史记录
+ (BOOL)isAdvancedAIFeaturesEnabled;        // 高级 AI 功能

@end
```

## 十三、总结

### 13.1 架构优势

1. **最小侵入性**: 主要修改集中在 EN4NotesModule 层面，不影响其他模块的稳定性
2. **模块化设计**: AI Area 作为独立模块，便于开发、测试和维护
3. **灵活布局**: 通过 NSSplitView 实现用户可自定义的布局比例
4. **响应式设计**: 适应不同窗口大小和使用场景
5. **性能优化**: 采用懒加载和合理的资源管理策略

### 13.2 用户体验亮点

1. **直观操作**: 拖拽调整、快捷键切换等便捷的交互方式
2. **上下文感知**: AI 功能与当前笔记内容深度集成
3. **多模式支持**: 聊天、工具、历史等多种 AI 交互模式
4. **状态保持**: 记住用户的布局偏好和使用习惯
5. **平滑动画**: 流畅的显示/隐藏和模式切换动画

### 13.3 技术实现特色

1. **分割视图**: 利用 macOS 原生 NSSplitView 实现专业的布局管理
2. **委托模式**: 通过协议实现组件间的松耦合通信
3. **状态管理**: 完整的状态持久化和恢复机制
4. **通知系统**: 基于 NSNotificationCenter 的事件驱动架构
5. **内存优化**: 智能的懒加载和资源清理策略

### 13.4 扩展性考虑

这种设计为未来的功能扩展提供了坚实的基础：

- **新 AI 功能**: 可以轻松添加新的 AI 工具和交互模式
- **界面定制**: 支持用户自定义 AI 区域的外观和行为
- **第三方集成**: 为第三方 AI 服务集成预留了接口
- **多语言支持**: 架构支持国际化和本地化扩展

通过这种精心设计的架构，Evernote Mac 应用可以无缝集成强大的 AI 功能，同时保持原有的稳定性和用户体验，为用户提供更加智能和高效的笔记管理体验。