# Evernote Mac 应用窗口布局自适应系统文档

## 概述

Evernote Mac 应用采用多级嵌套的 NSSplitView 架构来实现窗口布局的自适应调整。在窗口缩放过程中，各个区域（Sidebar、Note List、Editor View、AI Area）会根据优先级和约束规则进行动态调整，以确保最佳的用户体验。

## 布局结构

```
┌─ 主窗口 ─────────────────────────────────────────────────────┐
│ ┌─ Sidebar ─┐ ┌─ 主内容区域 ─────────────────────────────────┐ │
│ │           │ │ ┌─ Note List ─┐ ┌─ Editor + AI Area ──────┐ │ │
│ │  Navigation │ │ │           │ │ ┌─ Editor ─┐ ┌─ AI ──┐ │ │ │
│ │    Area     │ │ │  Note     │ │ │         │ │ Area │ │ │ │
│ │           │ │ │  Collection │ │ │ Content │ │      │ │ │ │
│ │           │ │ │           │ │ │         │ │      │ │ │ │
│ └───────────┘ │ └───────────┘ │ └─────────┘ └──────┘ │ │ │
│               └─────────────────────────────────────────────┘ │
└─────────────────────────────────────────────────────────────┘
```

## 区域尺寸约束

### 关键常量定义

```objc
// Sidebar 相关
static const CGFloat kENNoteCollectionWindowGridNavigationWidth = 64.f;        // Narrow模式宽度
static const CGFloat kENNoteCollectionWindowOutlineNavigationMinWidth = 140.f; // Outline最小宽度
static const CGFloat kENNoteCollectionWindowOutlineNavigationMaxWidth = 300.f; // Outline最大宽度
static const CGFloat kENNoteCollectionWindowGridNavigationSnapWidth = 105.f;   // 切换阈值

// 主内容区域
static const CGFloat kENNoteCollectionWindowModuleMinimumWidth = 680.f;        // 主内容区最小宽度

// Note List
static const CGFloat minNoteListWidth = 180.f;                                 // Note List最小宽度

// Editor View
static const CGFloat kMinEditorWidth = 400.f;                                  // Editor最小宽度

// AI Area
static const CGFloat kAIAreaMinWidth = 200.f;                                  // AI区域最小宽度
static const CGFloat kAIAreaMaxWidth = 600.f;                                  // AI区域最大宽度
static const CGFloat kAIAreaDefaultWidth = 320.f;                              // AI区域默认宽度
```

## 自适应调整机制

### 1. 窗口级别调整（ENSplitViewController）

**触发条件**：窗口大小变化（`windowDidResize` 通知）

**调整逻辑**：
```objc
CGFloat windowWidth = 窗口总宽度;
CGFloat availableWidth = windowWidth - kENNoteCollectionWindowModuleMinimumWidth; // 可用于Sidebar的空间
CGFloat outlineThreshold = 220.0f; // 切换阈值（包含缓冲区）

if (当前是narrow模式 && availableWidth > outlineThreshold) {
    // 空间充足，切换到outline模式
    切换到outline模式();
    设置宽度为 MIN(availableWidth, userPreferredWidth);
} else if (当前是outline模式 && availableWidth <= outlineThreshold) {
    // 空间不足，强制切换到narrow模式
    切换到narrow模式();
    设置宽度为 64px;
}
```

### 2. 主内容区域调整（AI Container）

**触发条件**：AI区域分割线拖拽

**约束逻辑**：
```objc
CGFloat totalWidth = 主内容区域总宽度;
CGFloat aiAreaWidth = totalWidth - proposedPosition;

// 确保Note List + Editor View有足够空间
CGFloat minModuleContentWidth = 580.f; // 180px(NoteList) + 400px(Editor)
CGFloat maxAIAreaWidth = totalWidth - minModuleContentWidth;

// 限制AI区域宽度范围
maxAIAreaWidth = MIN(maxAIAreaWidth, kAIAreaMaxWidth); // 最大600px
CGFloat minAIAreaWidth = isAIAreaVisible ? kAIAreaMinWidth : 0; // 显示时最小200px

// 应用约束
if (aiAreaWidth > maxAIAreaWidth) {
    constrainedPosition = totalWidth - maxAIAreaWidth;
} else if (aiAreaWidth < minAIAreaWidth) {
    constrainedPosition = totalWidth - minAIAreaWidth;
}
```

### 3. Note List 区域保护（EN4NotesModule）

**触发条件**：Note List与Editor之间的分割线拖拽

**保护逻辑**：
```objc
CGFloat minNoteListWidth = 180.f;
CGFloat maxPosition = splitView.frame.size.width - kMinEditorWidth;

if (proposedPosition < minNoteListWidth) {
    // 保护Note List最小宽度
    return minNoteListWidth;
} else if (proposedPosition > maxPosition) {
    // 保护Editor最小宽度
    return maxPosition;
}
```

## 窗口缩放行为分析

### 场景一：大窗口缩小过程

**初始状态**：1600px窗口，Sidebar为outline模式（300px）

1. **1600px → 1000px**：
   - Sidebar保持outline模式
   - 主内容区域从1300px缩小到700px
   - Note List和Editor按比例调整

2. **1000px → 900px**：
   - `availableWidth = 900 - 680 = 220px`
   - 刚好达到阈值，Sidebar切换到narrow模式（64px）
   - 主内容区域从700px增加到836px

3. **900px → 870px（最小窗口）**：
   - Sidebar保持narrow模式
   - 主内容区域从836px缩小到806px
   - 触发AI区域约束，确保Note List不被过度压缩

### 场景二：小窗口扩大过程

**初始状态**：870px窗口，Sidebar为narrow模式（64px）

1. **870px → 900px**：
   - `availableWidth = 900 - 680 = 220px`
   - 刚好达到阈值，但不立即切换（防止闪烁）

2. **900px → 920px**：
   - `availableWidth = 920 - 680 = 240px > 220px`
   - Sidebar切换到outline模式，宽度设为用户偏好值

3. **920px → 1600px**：
   - Sidebar保持outline模式
   - 各区域按比例扩大

### 场景三：AI区域调整

**触发**：用户拖拽AI区域分割线

1. **向左拖拽（扩大AI区域）**：
   - AI区域宽度增加，但不超过600px
   - 当总空间不足时，优先压缩Editor，但保证Editor不小于400px
   - Note List始终保持最小180px不变

2. **向右拖拽（缩小AI区域）**：
   - AI区域宽度减少，但显示时不小于200px
   - 释放的空间优先给Editor使用

## 优先级和约束关系

### 调整优先级（从高到低）

1. **Note List最小宽度**（180px）- 绝对保护
2. **Editor最小宽度**（400px）- 绝对保护
3. **Sidebar模式切换**（narrow ↔ outline）- 响应空间变化
4. **AI区域调整**（200-600px）- 在剩余空间内调整
5. **比例调整** - 在满足最小约束后按比例分配

### 约束冲突解决

当窗口过小无法满足所有最小宽度要求时：

1. **隐藏AI区域** - 释放200-600px空间
2. **强制narrow模式** - Sidebar占用最小64px
3. **维持核心功能** - 确保Note List（180px）+ Editor（400px）可用

## 性能和用户体验优化

### 防闪烁机制

- **阈值缓冲区**：narrow→outline使用220px，outline→narrow使用220px，避免临界值附近反复切换
- **统一判断标准**：两个方向的切换都基于`availableWidth`，确保逻辑一致性

### 响应性优化

- **实时约束检查**：在分割线拖拽过程中实时应用约束
- **平滑过渡**：使用动画减少突兀的尺寸跳变
- **状态持久化**：记住用户的布局偏好，重启后恢复

### 调试支持

- **详细日志**：使用`[SIDEBAR_DEBUG]`和`[SPLIT_DEBUG]`前缀记录关键决策点
- **状态追踪**：记录每次调整的原因和结果
- **性能监控**：避免在resize过程中进行昂贵的计算

## 总结

Evernote Mac的窗口布局系统通过多层级的约束和优先级机制，实现了在各种窗口尺寸下的最佳用户体验。核心设计原则是：

1. **保护核心功能**：Note List和Editor始终保持可用的最小尺寸
2. **智能适应**：Sidebar根据可用空间自动切换显示模式
3. **用户控制**：在约束范围内最大化用户的布局自定义能力
4. **性能优先**：避免不必要的重绘和计算，确保流畅的交互体验

这套系统确保了无论在什么尺寸的屏幕上，用户都能获得合理的界面布局和流畅的使用体验。