# Whiteboard WebView 跨域问题解决方案

## 问题描述

在 Evernote Mac 应用的 Whiteboard 功能中，遇到以下跨域问题：

- `client.html` 通过 `file://` 协议加载
- JavaScript 中的 `fetch()` 请求访问 `evernotecid://` 协议资源
- 浏览器安全策略阻止了从 `file://` 到 `evernotecid://` 的跨域请求

## 解决方案：JavaScript拦截方案

### 核心思路

通过在 WebView 中注入 JavaScript 代码拦截网络请求，确保 `evernotecid://` 协议请求能够正确发出并通过现有的 `ContentProtocolDownloader` 机制获取数据。

### 方案架构

```
┌─────────────────┐    ┌──────────────────┐    ┌────────────────────┐
│  file://client  │    │  JS Interceptor  │    │ ContentProtocol    │
│     .html       │───▶│                  │───▶│   Downloader       │
└─────────────────┘    └──────────────────┘    └────────────────────┘
                              │                           │
                              ▼                           ▼
                       ┌──────────────┐          ┌─────────────────┐
                       │ CORS Policy  │          │ ENContentProtocol│
                       │   Handling   │          │   (Data Source) │
                       └──────────────┘          └─────────────────┘
```

### 数据流程

1. JavaScript: `fetch('evernotecid://...')` (发起请求)
2. [JavaScript拦截器] (设置正确的CORS参数)
3. WKURLSchemeHandler (ContentProtocolDownloader) (处理evernotecid请求)
4. ENContentProtocol (实际获取数据)
5. 返回buffer数据给fetch() (标准Web API响应)
6. JavaScript: `response.arrayBuffer()` (获得数据)

## 实现方案

### 1. 主要拦截器实现

```objc
// ENWhiteboardEditorWKWebView.m

#pragma mark - Network Interceptor Setup

- (void)setupNetworkInterceptor {
    // 1. 创建拦截器脚本
    NSString *interceptorScript = [self createNetworkInterceptorScript];
    
    // 2. 注入到WebView中
    WKUserScript *userScript = [[WKUserScript alloc] initWithSource:interceptorScript
                                                      injectionTime:WKUserScriptInjectionTimeAtDocumentStart
                                                   forMainFrameOnly:NO];
    
    [self.configuration.userContentController addUserScript:userScript];
    
    // 3. 添加调试消息处理器
    [self.configuration.userContentController addScriptMessageHandler:self name:@"whiteboardDebug"];
    
    AppLogInfo(@"[Whiteboard] Network interceptor setup completed");
}
```

### 2. JavaScript拦截脚本

```javascript
(function() {
    'use strict';
    
    // ============= 配置部分 =============
    const CONFIG = {
        targetScheme: 'evernotecid',
        debug: true,
        logPrefix: '[Whiteboard Interceptor]'
    };
    
    // ============= Fetch API 拦截 =============
    const originalFetch = window.fetch;
    
    window.fetch = function(input, init) {
        let url = typeof input === 'string' ? input : (input.url || '');
        let isTargetRequest = url.startsWith(CONFIG.targetScheme + '://');
        
        if (isTargetRequest) {
            debugLog('Fetch request intercepted:', {
                url: url,
                method: (init && init.method) || 'GET'
            });
            
            // 关键：为evernotecid请求设置正确的CORS策略
            let finalInit = {
                ...init,
                mode: 'cors',           // 告诉浏览器这是CORS请求
                credentials: 'same-origin', // 合适的凭证策略
                cache: 'default'        // 缓存策略
            };
            
            if (typeof input !== 'string' && input instanceof Request) {
                // 重新创建Request对象
                input = new Request(url, {
                    method: input.method,
                    headers: input.headers,
                    body: input.body,
                    mode: 'cors',
                    credentials: input.credentials || 'same-origin',
                    cache: input.cache || 'default',
                    redirect: input.redirect || 'follow',
                    ...finalInit
                });
                finalInit = undefined;
            }
        }
        
        // 执行原始fetch
        return originalFetch.call(this, input, finalInit || init);
    };
    
    debugLog('Network interceptor initialized successfully');
})();
```

### 3. 消息处理器

```objc
#pragma mark - WKScriptMessageHandler

- (void)userContentController:(WKUserContentController *)userContentController 
      didReceiveScriptMessage:(WKScriptMessage *)message {
    
    if ([message.name isEqualToString:@"whiteboardDebug"]) {
        NSDictionary *debugData = message.body;
        NSString *level = debugData[@"level"] ?: @"info";
        NSString *msg = debugData[@"message"] ?: @"";
        NSString *data = debugData[@"data"] ?: @"";
        
        if ([level isEqualToString:@"error"]) {
            AppLogError(@"[Whiteboard JS] %@: %@", msg, data);
        } else {
            AppLogInfo(@"[Whiteboard JS] %@: %@", msg, data);
        }
    }
}
```

### 4. 初始化集成

```objc
#pragma mark - Integration

- (instancetype)initWithAccountController:(ENAccountController *)accountController {
    if (self = [super initWithAccountController:accountController]) {
        // 在WebView初始化时设置拦截器
        [self setupNetworkInterceptor];
    }
    return self;
}
```

## 关键技术要点

### 1. CORS参数设置

```javascript
// 关键设置
finalInit = {
    mode: 'cors',           // 明确告诉WebKit这是CORS请求
    credentials: 'same-origin', // 合适的凭证策略
    cache: 'default'        // 缓存策略
};
```

**为什么有效**：
- `mode: 'cors'` 明确告诉 WebKit 这是一个跨域请求，应该按照 CORS 规则处理
- `ContentProtocolDownloader` 作为 `WKURLSchemeHandler` 可以响应这种请求
- WebKit 会正确地将响应传递回 JavaScript

### 2. 请求处理流程

1. **拦截识别**：检查URL是否为 `evernotecid://` 协议
2. **参数设置**：为跨域请求设置正确的CORS参数
3. **原生处理**：通过现有的 `ContentProtocolDownloader` 获取数据
4. **响应返回**：WebKit包装成标准HTTP响应返回给JavaScript

### 3. 数据获取示例

前端JavaScript中可以正常使用：

```javascript
// 在client.html中，这样的代码现在应该能正常工作：
async function loadImageBuffer() {
    try {
        const response = await fetch('evernotecid://84D5E09F-B16A-446E-9E00-DB3B80E615C1/stage4yinxiangcom/10241/ENResource/p16');
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        // 获取buffer数据 - 这是真实的图片数据
        const buffer = await response.arrayBuffer();
        
        console.log('Image loaded successfully:', {
            size: buffer.byteLength,
            type: response.headers.get('content-type')
        });
        
        return buffer;
        
    } catch (error) {
        console.error('Failed to load image:', error);
        throw error;
    }
}
```

## 方案优势

1. **最小侵入性**：无需修改现有的 `client.html` 或其他前端代码
2. **利用现有基础设施**：完全复用 `ContentProtocolDownloader` 和 `ENContentProtocol`
3. **标准兼容**：使用标准的 Web API，不依赖私有接口
4. **易于调试**：提供详细的日志和测试机制
5. **渐进式部署**：可以先在开发环境测试，确认有效后再部署

## 潜在风险

1. **WebKit版本依赖**：不同版本的WebKit对CORS处理可能有差异
2. **性能开销**：每个请求都需要经过JavaScript处理
3. **维护复杂度**：需要维护JavaScript拦截逻辑

## 测试验证

### 测试脚本

```objc
- (void)testNetworkInterceptor {
    NSString *testScript = @`
        const testURL = 'evernotecid://84D5E09F-B16A-446E-9E00-DB3B80E615C1/stage4yinxiangcom/10241/ENResource/p16';
        
        fetch(testURL)
            .then(response => {
                console.log('[Test] Fetch test result:', {
                    ok: response.ok,
                    status: response.status,
                    contentType: response.headers.get('content-type')
                });
                return response.arrayBuffer();
            })
            .then(buffer => {
                console.log('[Test] Buffer received:', {
                    size: buffer.byteLength
                });
            })
            .catch(error => {
                console.error('[Test] Fetch test failed:', error);
            });
    `;
    
    [self evaluateJavaScript:testScript completionHandler:nil];
}
```

## 实施步骤

1. **第一步**：实现基础版本，验证核心功能
2. **第二步**：添加错误处理和重试机制
3. **第三步**：优化性能和用户体验
4. **第四步**：全面测试和部署

## 结论

JavaScript拦截方案通过最小化的代码修改，利用现有的成熟基础设施，有效解决了Whiteboard WebView的跨域问题。该方案不仅技术可行，而且实施风险较低，是当前最推荐的解决方案。