//
//  ENWhiteboardEditorWKWebView_Implementation.m
//  Evernote
//
//  Implementation for Whiteboard WebView Cross-Origin Solution
//  解决 Whiteboard WebView 跨域问题的具体实现
//

#import "ENWhiteboardEditorWKWebView_Implementation.h"
#import "ENCommonEditorWKWebView_Private.h"

@import ENLogging;
@import WebKit;

#pragma mark - Configuration Constants

NSString * const ENWhiteboardInterceptorTargetScheme = @"evernotecid";
NSString * const ENWhiteboardInterceptorLogPrefix = @"[Whiteboard Interceptor]";
NSString * const ENWhiteboardInterceptorDebugMessageHandler = @"whiteboardDebug";

NSInteger const ENWhiteboardInterceptorDefaultTimeout = 30000;  // 30秒
NSInteger const ENWhiteboardInterceptorMaxRetries = 3;
NSInteger const ENWhiteboardInterceptorRetryDelay = 1000;       // 1秒

#pragma mark - Notification Names

NSString * const ENWhiteboardInterceptorDidSetupNotification = @"ENWhiteboardInterceptorDidSetupNotification";
NSString * const ENWhiteboardInterceptorDidReceiveRequestNotification = @"ENWhiteboardInterceptorDidReceiveRequestNotification";
NSString * const ENWhiteboardInterceptorDidCompleteRequestNotification = @"ENWhiteboardInterceptorDidCompleteRequestNotification";
NSString * const ENWhiteboardInterceptorDidFailRequestNotification = @"ENWhiteboardInterceptorDidFailRequestNotification";

#pragma mark - Implementation

@implementation ENWhiteboardEditorWKWebView (NetworkInterceptor)

#pragma mark - Lifecycle Integration

- (instancetype)initWithAccountController:(ENAccountController *)accountController {
    if (self = [super initWithAccountController:accountController]) {
        // 在WebView初始化时设置拦截器
        [self setupNetworkInterceptor];
    }
    return self;
}

#pragma mark - Network Interceptor Setup

- (void)setupNetworkInterceptor {
    AppLogInfo(@"%@ Setting up network interceptor...", ENWhiteboardInterceptorLogPrefix);
    
    // 1. 创建拦截器脚本
    NSString *interceptorScript = [self createNetworkInterceptorScript];
    
    // 2. 注入到WebView中
    WKUserScript *userScript = [[WKUserScript alloc] initWithSource:interceptorScript
                                                      injectionTime:WKUserScriptInjectionTimeAtDocumentStart
                                                   forMainFrameOnly:NO];
    
    [self.configuration.userContentController addUserScript:userScript];
    
    // 3. 添加调试消息处理器
    [self.configuration.userContentController addScriptMessageHandler:self 
                                                                  name:ENWhiteboardInterceptorDebugMessageHandler];
    
    AppLogInfo(@"%@ Network interceptor setup completed", ENWhiteboardInterceptorLogPrefix);
    
    // 4. 发送通知
    [[NSNotificationCenter defaultCenter] postNotificationName:ENWhiteboardInterceptorDidSetupNotification
                                                        object:self
                                                      userInfo:@{@"timestamp": [NSDate date]}];
}

- (NSString *)createNetworkInterceptorScript {
    NSString *script = [NSString stringWithFormat:@`
        (function() {
            'use strict';
            
            // ============= 配置部分 =============
            const CONFIG = {
                targetScheme: '%@',
                debug: true,
                logPrefix: '%@',
                timeout: %ld,
                maxRetries: %ld,
                retryDelay: %ld
            };
            
            // ============= 工具函数 =============
            function debugLog(message, data) {
                if (CONFIG.debug) {
                    console.log(CONFIG.logPrefix, message, data || '');
                    
                    // 发送调试信息到原生代码
                    if (window.webkit && window.webkit.messageHandlers && window.webkit.messageHandlers.%@) {
                        try {
                            window.webkit.messageHandlers.%@.postMessage({
                                level: 'info',
                                message: message,
                                data: JSON.stringify(data || {}),
                                timestamp: new Date().toISOString()
                            });
                        } catch (e) {
                            console.warn('Failed to send debug message to native:', e);
                        }
                    }
                }
            }
            
            function debugError(message, error) {
                console.error(CONFIG.logPrefix, message, error);
                
                if (window.webkit && window.webkit.messageHandlers && window.webkit.messageHandlers.%@) {
                    try {
                        window.webkit.messageHandlers.%@.postMessage({
                            level: 'error',
                            message: message,
                            data: JSON.stringify({
                                error: error.message || error.toString(),
                                stack: error.stack
                            }),
                            timestamp: new Date().toISOString()
                        });
                    } catch (e) {
                        console.warn('Failed to send error message to native:', e);
                    }
                }
            }
            
            function isEvernoteCidURL(url) {
                return typeof url === 'string' && url.startsWith(CONFIG.targetScheme + '://');
            }
            
            // ============= 重试机制 =============
            async function fetchWithRetry(input, init, retryCount = 0) {
                const url = typeof input === 'string' ? input : input.url;
                
                try {
                    // 设置超时控制
                    const controller = new AbortController();
                    const timeoutId = setTimeout(() => {
                        controller.abort();
                        debugError('Request timeout', new Error('Request timeout after ' + CONFIG.timeout + 'ms'));
                    }, CONFIG.timeout);
                    
                    const finalInit = {
                        ...init,
                        signal: controller.signal
                    };
                    
                    debugLog('Attempting fetch request:', {
                        url: url,
                        method: finalInit.method || 'GET',
                        retry: retryCount,
                        timeout: CONFIG.timeout
                    });
                    
                    const response = await originalFetch.call(window, input, finalInit);
                    clearTimeout(timeoutId);
                    
                    if (!response.ok && retryCount < CONFIG.maxRetries) {
                        debugLog('Request failed, retrying:', {
                            status: response.status,
                            statusText: response.statusText,
                            url: url,
                            retryCount: retryCount + 1,
                            maxRetries: CONFIG.maxRetries
                        });
                        
                        await new Promise(resolve => setTimeout(resolve, CONFIG.retryDelay));
                        return await fetchWithRetry(input, init, retryCount + 1);
                    }
                    
                    debugLog('Request completed successfully:', {
                        url: url,
                        status: response.status,
                        contentType: response.headers.get('content-type'),
                        contentLength: response.headers.get('content-length')
                    });
                    
                    return response;
                    
                } catch (error) {
                    if (error.name === 'AbortError') {
                        const timeoutError = new Error('Request timeout after ' + CONFIG.timeout + 'ms');
                        if (retryCount < CONFIG.maxRetries) {
                            debugLog('Request timeout, retrying:', {
                                url: url,
                                retryCount: retryCount + 1,
                                maxRetries: CONFIG.maxRetries
                            });
                            await new Promise(resolve => setTimeout(resolve, CONFIG.retryDelay));
                            return await fetchWithRetry(input, init, retryCount + 1);
                        }
                        throw timeoutError;
                    }
                    
                    if (retryCount < CONFIG.maxRetries) {
                        debugLog('Request error, retrying:', {
                            error: error.message,
                            url: url,
                            retryCount: retryCount + 1,
                            maxRetries: CONFIG.maxRetries
                        });
                        
                        await new Promise(resolve => setTimeout(resolve, CONFIG.retryDelay));
                        return await fetchWithRetry(input, init, retryCount + 1);
                    }
                    
                    debugError('Request failed after all retries:', error);
                    throw error;
                }
            }
            
            // ============= Fetch API 拦截 =============
            const originalFetch = window.fetch;
            
            window.fetch = function(input, init) {
                let url = typeof input === 'string' ? input : (input.url || '');
                let isTargetRequest = isEvernoteCidURL(url);
                
                if (isTargetRequest) {
                    debugLog('Fetch request intercepted:', {
                        url: url,
                        method: (init && init.method) || 'GET',
                        headers: init && init.headers ? Object.fromEntries(new Headers(init.headers)) : {}
                    });
                }
                
                let finalInput = input;
                let finalInit = init || {};
                
                if (isTargetRequest) {
                    // 关键：为evernotecid请求设置正确的CORS策略
                    if (typeof input === 'string') {
                        finalInput = url;
                        finalInit = {
                            ...finalInit,
                            mode: 'cors',              // 告诉浏览器这是CORS请求
                            credentials: 'same-origin', // 合适的凭证策略
                            cache: finalInit.cache || 'default',
                            redirect: finalInit.redirect || 'follow'
                        };
                    } else if (input instanceof Request) {
                        // 重新创建Request对象，确保设置正确
                        finalInput = new Request(url, {
                            method: input.method,
                            headers: input.headers,
                            body: input.body,
                            mode: 'cors',  // 关键设置
                            credentials: input.credentials || 'same-origin',
                            cache: input.cache || 'default',
                            redirect: input.redirect || 'follow',
                            referrer: input.referrer,
                            referrerPolicy: input.referrerPolicy,
                            integrity: input.integrity,
                            keepalive: input.keepalive,
                            signal: input.signal,
                            ...finalInit  // 合并额外的init参数
                        });
                        finalInit = undefined; // Request对象已包含所有配置
                    }
                    
                    // 使用重试机制处理evernotecid请求
                    return fetchWithRetry(finalInput, finalInit);
                } else {
                    // 非目标请求直接使用原始fetch
                    return originalFetch.call(this, finalInput, finalInit);
                }
            };
            
            // ============= XMLHttpRequest 拦截 =============
            const OriginalXMLHttpRequest = window.XMLHttpRequest;
            
            function InterceptedXMLHttpRequest() {
                const xhr = new OriginalXMLHttpRequest();
                const originalOpen = xhr.open;
                
                xhr.open = function(method, url, async, user, password) {
                    if (isEvernoteCidURL(url)) {
                        debugLog('XHR request intercepted:', {
                            method: method,
                            url: url,
                            async: async
                        });
                    }
                    
                    return originalOpen.call(this, method, url, async, user, password);
                };
                
                return xhr;
            }
            
            // 保持原型链
            InterceptedXMLHttpRequest.prototype = OriginalXMLHttpRequest.prototype;
            Object.setPrototypeOf(InterceptedXMLHttpRequest, OriginalXMLHttpRequest);
            
            // 替换全局XMLHttpRequest
            window.XMLHttpRequest = InterceptedXMLHttpRequest;
            
            // ============= 初始化完成 =============
            debugLog('Network interceptor initialized successfully', {
                userAgent: navigator.userAgent,
                location: window.location.href,
                timestamp: new Date().toISOString(),
                config: CONFIG
            });
            
            // 为调试提供全局访问
            window.__whiteboardInterceptor = {
                config: CONFIG,
                debugLog: debugLog,
                debugError: debugError,
                isEvernoteCidURL: isEvernoteCidURL,
                originalFetch: originalFetch,
                fetchWithRetry: fetchWithRetry,
                version: '1.0.0'
            };
            
        })();
    `, 
    ENWhiteboardInterceptorTargetScheme,
    ENWhiteboardInterceptorLogPrefix,
    (long)ENWhiteboardInterceptorDefaultTimeout,
    (long)ENWhiteboardInterceptorMaxRetries,
    (long)ENWhiteboardInterceptorRetryDelay,
    ENWhiteboardInterceptorDebugMessageHandler,
    ENWhiteboardInterceptorDebugMessageHandler,
    ENWhiteboardInterceptorDebugMessageHandler,
    ENWhiteboardInterceptorDebugMessageHandler];
    
    return script;
}

- (NSString *)createAdvancedInterceptorScript {
    // 这里可以实现更高级的功能，如请求缓存、批量处理等
    // 当前返回基础版本
    return [self createNetworkInterceptorScript];
}

#pragma mark - WKScriptMessageHandler

- (void)userContentController:(WKUserContentController *)userContentController 
      didReceiveScriptMessage:(WKScriptMessage *)message {
    
    if ([message.name isEqualToString:ENWhiteboardInterceptorDebugMessageHandler]) {
        NSDictionary *debugData = message.body;
        NSString *level = debugData[@"level"] ?: @"info";
        NSString *msg = debugData[@"message"] ?: @"";
        NSString *data = debugData[@"data"] ?: @"";
        NSString *timestamp = debugData[@"timestamp"] ?: @"";
        
        NSString *logMessage = [NSString stringWithFormat:@"%@ %@: %@", ENWhiteboardInterceptorLogPrefix, msg, data];
        
        if ([level isEqualToString:@"error"]) {
            AppLogError(@"%@", logMessage);
            
            // 发送错误通知
            [[NSNotificationCenter defaultCenter] postNotificationName:ENWhiteboardInterceptorDidFailRequestNotification
                                                                object:self
                                                              userInfo:@{
                                                                  @"message": msg,
                                                                  @"data": data,
                                                                  @"timestamp": timestamp
                                                              }];
        } else {
            AppLogInfo(@"%@", logMessage);
            
            // 检查是否是请求相关的消息
            if ([msg containsString:@"request"] || [msg containsString:@"fetch"]) {
                NSString *notificationName = [msg containsString:@"completed"] ? 
                    ENWhiteboardInterceptorDidCompleteRequestNotification : 
                    ENWhiteboardInterceptorDidReceiveRequestNotification;
                
                [[NSNotificationCenter defaultCenter] postNotificationName:notificationName
                                                                    object:self
                                                                  userInfo:@{
                                                                      @"message": msg,
                                                                      @"data": data,
                                                                      @"timestamp": timestamp
                                                                  }];
            }
        }
    }
}

#pragma mark - Testing and Validation

- (void)testNetworkInterceptor {
    AppLogInfo(@"%@ Starting network interceptor test...", ENWhiteboardInterceptorLogPrefix);
    
    NSString *testScript = @`
        (function() {
            console.log('[Test] Starting network interceptor test...');
            
            // 测试URL - 使用一个典型的evernotecid资源URL
            const testURL = 'evernotecid://84D5E09F-B16A-446E-9E00-DB3B80E615C1/stage4yinxiangcom/10241/ENResource/p16';
            
            // 检查拦截器是否已安装
            if (!window.__whiteboardInterceptor) {
                console.error('[Test] Interceptor not found!');
                return;
            }
            
            console.log('[Test] Interceptor found, version:', window.__whiteboardInterceptor.version);
            
            // 执行测试请求
            fetch(testURL)
                .then(response => {
                    console.log('[Test] Fetch test result:', {
                        ok: response.ok,
                        status: response.status,
                        statusText: response.statusText,
                        contentType: response.headers.get('content-type'),
                        contentLength: response.headers.get('content-length')
                    });
                    
                    if (response.ok) {
                        return response.arrayBuffer();
                    } else {
                        throw new Error('Response not ok: ' + response.status + ' ' + response.statusText);
                    }
                })
                .then(buffer => {
                    console.log('[Test] Buffer received:', {
                        size: buffer.byteLength,
                        constructor: buffer.constructor.name
                    });
                    
                    // 发送测试结果到原生代码
                    if (window.webkit && window.webkit.messageHandlers && window.webkit.messageHandlers.whiteboardDebug) {
                        window.webkit.messageHandlers.whiteboardDebug.postMessage({
                            level: 'info',
                            message: 'Test completed successfully',
                            data: JSON.stringify({ 
                                bufferSize: buffer.byteLength,
                                testURL: testURL,
                                success: true 
                            }),
                            timestamp: new Date().toISOString()
                        });
                    }
                })
                .catch(error => {
                    console.error('[Test] Fetch test failed:', error);
                    
                    if (window.webkit && window.webkit.messageHandlers && window.webkit.messageHandlers.whiteboardDebug) {
                        window.webkit.messageHandlers.whiteboardDebug.postMessage({
                            level: 'error',
                            message: 'Test failed',
                            data: JSON.stringify({ 
                                error: error.message,
                                testURL: testURL,
                                success: false 
                            }),
                            timestamp: new Date().toISOString()
                        });
                    }
                });
        })();
    `;
    
    [self evaluateJavaScript:testScript completionHandler:^(id result, NSError *error) {
        if (error) {
            AppLogError(@"%@ Test script execution failed: %@", ENWhiteboardInterceptorLogPrefix, error);
        } else {
            AppLogInfo(@"%@ Test script executed successfully", ENWhiteboardInterceptorLogPrefix);
        }
    }];
}

- (void)validateEvernoteCidRequest:(NSString *)testURL {
    if (!testURL || ![testURL hasPrefix:ENWhiteboardInterceptorTargetScheme]) {
        AppLogError(@"%@ Invalid test URL: %@", ENWhiteboardInterceptorLogPrefix, testURL);
        return;
    }
    
    NSString *validationScript = [NSString stringWithFormat:@`
        fetch('%@')
            .then(response => response.arrayBuffer())
            .then(buffer => {
                console.log('Validation successful for URL: %@, buffer size:', buffer.byteLength);
            })
            .catch(error => {
                console.error('Validation failed for URL: %@, error:', error);
            });
    `, testURL, testURL, testURL];
    
    [self evaluateJavaScript:validationScript completionHandler:nil];
}

#pragma mark - Debug and Logging

- (void)setDebugLoggingEnabled:(BOOL)enabled {
    NSString *debugScript = [NSString stringWithFormat:@`
        if (window.__whiteboardInterceptor) {
            window.__whiteboardInterceptor.config.debug = %@;
            console.log('%@ Debug logging %@');
        }
    `, enabled ? @"true" : @"false", ENWhiteboardInterceptorLogPrefix, enabled ? @"enabled" : @"disabled"];
    
    [self evaluateJavaScript:debugScript completionHandler:nil];
}

- (NSDictionary *)getInterceptorStatus {
    // 这里可以返回拦截器的状态信息
    // 实际实现中可能需要通过JavaScript bridge获取动态状态
    return @{
        @"isSetup": @YES,
        @"targetScheme": ENWhiteboardInterceptorTargetScheme,
        @"version": @"1.0.0",
        @"timestamp": [NSDate date]
    };
}

#pragma mark - WebView Lifecycle

- (void)webView:(WKWebView *)webView didFinishNavigation:(WKNavigation *)navigation {
    [super webView:webView didFinishNavigation:navigation];
    
    // 页面加载完成后延迟执行测试
    dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(2.0 * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
        [self testNetworkInterceptor];
    });
}

@end