# Whiteboard HTTP服务器使用指南

## 📋 快速开始

### 基本使用

```objc
// 1. 创建HTTP服务器
ENWhiteboardHTTPServer *httpServer = [[ENWhiteboardHTTPServer alloc] 
    initWithAccountController:self.accountController];

// 2. 启动服务器
BOOL success = [httpServer startServer];
if (success) {
    NSLog(@"HTTP服务器启动成功: %@", [httpServer baseURL]);
} else {
    NSLog(@"HTTP服务器启动失败");
}

// 3. 获取URL并加载
NSURL *httpURL = [httpServer indexHTMLURL];
[webView loadRequest:[NSURLRequest requestWithURL:httpURL]];
```

### 远程服务器配置

```objc
// 配置远程HTML服务器
NSURL *remoteURL = [NSURL URLWithString:@"http://10.228.32.31:8888"];
[httpServer configureRemoteServer:remoteURL];

// 启动服务器
[httpServer startServer];

// 现在所有请求都会代理到远程服务器
NSURL *proxyURL = [httpServer indexHTMLURL];
// 例如: http://localhost:54321/index.html → http://10.228.32.31:8888/index.html
```

## 🔧 详细API参考

### ENWhiteboardHTTPServer类

#### 初始化方法
```objc
/**
 * 初始化HTTP服务器
 * @param accountController 账户控制器，用于获取evernotecid资源
 * @return HTTP服务器实例
 */
- (instancetype)initWithAccountController:(ENAccountController *)accountController;
```

#### 服务器控制
```objc
/**
 * 启动HTTP服务器
 * 自动分配可用端口并开始监听
 * @return YES表示启动成功，NO表示启动失败
 */
- (BOOL)startServer;

/**
 * 停止HTTP服务器
 * 清理所有资源和连接
 */
- (void)stopServer;

/**
 * 获取服务器运行状态
 * @return YES表示正在运行，NO表示已停止
 */
- (BOOL)isRunning;
```

#### URL生成
```objc
/**
 * 获取index.html的完整URL
 * @return HTTP服务器上index.html的URL
 */
- (nullable NSURL *)indexHTMLURL;

/**
 * 将evernotecid URL转换为HTTP API URL
 * @param evernoteCidURL evernotecid协议的URL
 * @return 对应的HTTP API URL
 */
- (nullable NSURL *)httpURLForEvernoteCidURL:(NSString *)evernoteCidURL;

/**
 * 获取服务器基础URL
 * @return 格式：http://localhost:PORT
 */
- (nullable NSURL *)baseURL;

/**
 * 获取服务器监听端口
 * @return 自动分配的可用端口
 */
- (NSUInteger)port;
```

#### 配置方法
```objc
/**
 * 配置远程HTML服务器
 * @param remoteURL 远程服务器的基础URL，例如：http://10.228.32.31:8888
 */
- (void)configureRemoteServer:(nullable NSURL *)remoteURL;

/**
 * 设置调试模式
 * @param enabled YES启用调试日志，NO关闭
 */
- (void)setDebugEnabled:(BOOL)enabled;

/**
 * 获取服务器状态信息
 * @return 包含端口、运行状态等信息的字典
 */
- (NSDictionary *)serverStatus;
```

#### 属性
```objc
// 只读属性
@property (nonatomic, readonly) NSUInteger port;
@property (nonatomic, readonly, nullable) NSURL *baseURL;
@property (nonatomic, readonly, getter=isRunning) BOOL running;
@property (nonatomic, readonly, nullable) NSString *resourcePath;

// 读写属性
@property (nonatomic, weak, nullable) ENAccountController *accountController;
@property (nonatomic, strong, nullable) NSURL *remoteServerURL;
```

## 🌐 工作模式

### 1. 本地文件模式 (默认)

```objc
// 不设置远程服务器URL，使用本地文件
ENWhiteboardHTTPServer *server = [[ENWhiteboardHTTPServer alloc] 
    initWithAccountController:accountController];
[server startServer];

// 服务器将提供以下内容:
// http://localhost:PORT/index.html     → common-database-mac/index.html
// http://localhost:PORT/client.html    → common-database-mac/client.html  
// http://localhost:PORT/superTable.js  → common-database-mac/superTable.js
// http://localhost:PORT/css/xxx.css    → common-database-mac/css/xxx.css
```

#### 特点
- ✅ 快速响应，无网络延迟
- ✅ 离线工作
- ✅ 适合开发和测试
- ❌ 需要本地文件存在

### 2. 远程代理模式

```objc
// 设置远程服务器URL
ENWhiteboardHTTPServer *server = [[ENWhiteboardHTTPServer alloc] 
    initWithAccountController:accountController];
[server configureRemoteServer:[NSURL URLWithString:@"http://10.228.32.31:8888"]];
[server startServer];

// 服务器将代理以下内容:
// http://localhost:PORT/index.html     → http://10.228.32.31:8888/index.html
// http://localhost:PORT/client.html    → http://10.228.32.31:8888/client.html
// http://localhost:PORT/superTable.js  → http://10.228.32.31:8888/superTable.js
```

#### 特点
- ✅ 可以使用远程部署的HTML
- ✅ 便于更新和维护
- ✅ 支持CDN等优化
- ❌ 依赖网络连接
- ❌ 增加网络延迟

## 📊 状态监控

### 获取服务器状态
```objc
NSDictionary *status = [httpServer serverStatus];
NSLog(@"服务器状态: %@", status);

// 输出示例:
// {
//     "running" = 1;
//     "port" = 54321;
//     "baseURL" = "http://localhost:54321";
//     "resourcePath" = "/path/to/common-database-mac";
//     "debugEnabled" = 1;
//     "mode" = "remote_proxy";  // 或 "local_files"
//     "remoteServerURL" = "http://10.228.32.31:8888";
//     "timestamp" = "2023-xx-xx xx:xx:xx";
// }
```

### 监听服务器事件
```objc
// 监听服务器启动成功
[[NSNotificationCenter defaultCenter] addObserver:self
                                         selector:@selector(serverDidStart:)
                                             name:ENWhiteboardHTTPServerDidStartNotification
                                           object:httpServer];

// 监听服务器停止
[[NSNotificationCenter defaultCenter] addObserver:self
                                         selector:@selector(serverDidStop:)
                                             name:ENWhiteboardHTTPServerDidStopNotification
                                           object:httpServer];

// 监听请求处理
[[NSNotificationCenter defaultCenter] addObserver:self
                                         selector:@selector(serverDidReceiveRequest:)
                                             name:ENWhiteboardHTTPServerDidReceiveRequestNotification
                                           object:httpServer];
```

## 🔍 调试和测试

### 启用调试模式
```objc
[httpServer setDebugEnabled:YES];
// 启用后会输出详细的请求日志
```

### 直接浏览器访问
```objc
NSURL *baseURL = [httpServer baseURL];
NSLog(@"可以在浏览器中访问: %@", baseURL);

// 浏览器访问示例:
// http://localhost:54321/test.html      - 测试页面
// http://localhost:54321/index.html     - 主页面
// http://localhost:54321/api/evernotecid/test - API测试
```

### 测试URL重写功能
在浏览器开发者工具Console中执行:
```javascript
// 测试URL重写是否工作
fetch('evernotecid://test-resource-id/sample/path')
  .then(response => console.log('Success:', response))
  .catch(error => console.error('Error:', error));

// 应该能看到URL被自动转换为:
// /api/evernotecid/test-resource-id/sample/path
```

## 🎯 最佳实践

### 1. 服务器生命周期管理
```objc
@interface MyViewController : NSViewController
@property (nonatomic, strong) ENWhiteboardHTTPServer *httpServer;
@end

@implementation MyViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    
    // 创建并启动服务器
    self.httpServer = [[ENWhiteboardHTTPServer alloc] 
        initWithAccountController:self.accountController];
    [self.httpServer startServer];
}

- (void)dealloc {
    // 确保停止服务器释放资源
    [self.httpServer stopServer];
}

@end
```

### 2. 错误处理
```objc
BOOL success = [httpServer startServer];
if (!success) {
    NSLog(@"HTTP服务器启动失败，回退到文件URL模式");
    // 可以回退到原来的file://URL加载方式
    NSURL *fileURL = [self fallbackFileURL];
    [webView loadRequest:[NSURLRequest requestWithURL:fileURL]];
}
```

### 3. 性能优化
```objc
// 只创建一次服务器实例，重复使用
+ (ENWhiteboardHTTPServer *)sharedHTTPServer {
    static ENWhiteboardHTTPServer *sharedServer = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        // 注意：这里需要合适的accountController实例
        sharedServer = [[ENWhiteboardHTTPServer alloc] 
            initWithAccountController:[ENAccountController sharedController]];
    });
    return sharedServer;
}
```

### 4. 配置切换
```objc
// 根据环境切换本地/远程模式
- (void)configureServerForEnvironment:(NSString *)environment {
    if ([environment isEqualToString:@"development"]) {
        // 开发环境使用本地文件
        [self.httpServer configureRemoteServer:nil];
    } else if ([environment isEqualToString:@"testing"]) {
        // 测试环境使用测试服务器
        NSURL *testURL = [NSURL URLWithString:@"http://test.example.com:8080"];
        [self.httpServer configureRemoteServer:testURL];
    } else {
        // 生产环境使用生产服务器
        NSURL *prodURL = [NSURL URLWithString:@"http://prod.example.com"];
        [self.httpServer configureRemoteServer:prodURL];
    }
}
```

## ⚠️ 注意事项

### 1. 端口冲突
- 服务器使用0端口自动分配，一般不会冲突
- 如需要固定端口，可修改startServer方法

### 2. 网络安全
- 本地HTTP服务器只监听localhost，外部无法访问
- 远程代理请求会暴露给网络，注意隐私保护

### 3. 资源路径
- 确保common-database-mac目录存在且包含必要文件
- 远程服务器URL需要可访问且返回正确的Content-Type

### 4. 内存使用
- 代理大文件时注意内存使用
- 考虑实现流式传输以减少内存占用

### 5. 并发请求
- 服务器支持并发请求，但要注意资源竞争
- 大量并发请求可能影响性能

## 🚀 集成示例

### 完整的WebView控制器集成
```objc
@interface ENWhiteboardViewController : NSViewController
@property (nonatomic, strong) WKWebView *webView;
@property (nonatomic, strong) ENWhiteboardHTTPServer *httpServer;
@property (nonatomic, weak) ENAccountController *accountController;
@end

@implementation ENWhiteboardViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    
    // 初始化WebView
    [self setupWebView];
    
    // 初始化HTTP服务器
    [self setupHTTPServer];
    
    // 加载内容
    [self loadWhiteboardContent];
}

- (void)setupWebView {
    WKWebViewConfiguration *config = [[WKWebViewConfiguration alloc] init];
    self.webView = [[WKWebView alloc] initWithFrame:self.view.bounds 
                                      configuration:config];
    self.webView.autoresizingMask = NSViewWidthSizable | NSViewHeightSizable;
    [self.view addSubview:self.webView];
}

- (void)setupHTTPServer {
    self.httpServer = [[ENWhiteboardHTTPServer alloc] 
        initWithAccountController:self.accountController];
    
    // 根据需要配置远程服务器
    // [self.httpServer configureRemoteServer:[NSURL URLWithString:@"http://remote.server"]];
    
    BOOL success = [self.httpServer startServer];
    if (success) {
        NSLog(@"Whiteboard HTTP服务器启动成功: %@", [self.httpServer baseURL]);
    } else {
        NSLog(@"Whiteboard HTTP服务器启动失败");
    }
}

- (void)loadWhiteboardContent {
    NSURL *httpURL = [self.httpServer indexHTMLURL];
    if (httpURL) {
        NSURLRequest *request = [NSURLRequest requestWithURL:httpURL];
        [self.webView loadRequest:request];
    } else {
        NSLog(@"无法获取HTTP URL，HTTP服务器可能未启动");
    }
}

- (void)dealloc {
    [self.httpServer stopServer];
}

@end
```

这个使用指南涵盖了Whiteboard HTTP服务器的所有使用方法和最佳实践，可以帮助开发者快速集成和使用该功能。