# Whiteboard远程HTML服务器测试指南

## 📋 测试概述

本文档提供了完整的远程HTML服务器测试方法和代码示例，帮助验证远程代理功能是否正常工作。

## 🚀 快速测试

### 1. 基本远程服务器测试

```objc
// 在你的ViewController中添加测试方法
- (void)testRemoteWhiteboardServer {
    // 获取或创建Whiteboard WebView
    ENWhiteboardEditorWKWebView *webView = self.whiteboardWebView;
    
    // 测试远程服务器URL
    NSString *remoteServerURL = @"http://10.228.32.31:8888";
    
    AppLogInfo(@"🧪 开始测试远程服务器: %@", remoteServerURL);
    
    // 配置并加载远程服务器
    [webView loadRemoteWhiteboardFromURL:remoteServerURL];
    
    // 获取服务器状态
    NSDictionary *status = [webView.httpServer serverStatus];
    AppLogInfo(@"📊 服务器状态: %@", status);
}
```

### 2. 快捷键测试

在你的应用中添加快捷键支持：

```objc
- (void)viewDidLoad {
    [super viewDidLoad];
    
    // 添加键盘监听
    [self setupKeyboardShortcuts];
}

- (void)setupKeyboardShortcuts {
    // 添加Command+R快捷键测试远程服务器
    NSMenuItem *testRemoteItem = [[NSMenuItem alloc] initWithTitle:@"Test Remote Server" 
                                                            action:@selector(testRemoteWhiteboardServer) 
                                                     keyEquivalent:@"r"];
    testRemoteItem.keyEquivalentModifierMask = NSEventModifierFlagCommand;
    testRemoteItem.target = self;
    
    // 添加到菜单或作为快捷键处理
}
```

### 3. 控制台测试命令

可以在Xcode控制台中执行以下LLDB命令进行测试：

```lldb
# 1. 获取当前的WebView实例
po ENWhiteboardEditorWKWebView *webView = (ENWhiteboardEditorWKWebView *)[[[[NSApplication sharedApplication] mainWindow] contentViewController] whiteboardWebView]

# 2. 测试远程服务器配置
po [webView testRemoteServerConfiguration]

# 3. 查看服务器状态
po [webView.httpServer serverStatus]

# 4. 测试不同的远程URL
po [webView loadRemoteWhiteboardFromURL:@"http://example.com:8080"]
```

## 🔧 详细测试步骤

### 步骤1: 准备远程HTML服务器

在远程服务器 (10.228.32.31:8888) 上部署以下测试文件：

#### index.html
```html
<!DOCTYPE html>
<html>
<head>
    <title>远程Whiteboard测试</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .test-section { margin: 20px 0; padding: 15px; border: 1px solid #ccc; }
        .success { color: green; }
        .error { color: red; }
        .info { color: blue; }
    </style>
</head>
<body>
    <h1>🌐 远程HTML服务器测试页面</h1>
    
    <div class="test-section">
        <h2>📋 基本信息</h2>
        <p>当前页面URL: <span id="currentUrl"></span></p>
        <p>协议: <span id="protocol"></span></p>
        <p>主机: <span id="hostname"></span></p>
        <p>端口: <span id="port"></span></p>
    </div>
    
    <div class="test-section">
        <h2>🔄 URL重写测试</h2>
        <p>测试evernotecid://协议是否被正确重写</p>
        <button onclick="testURLRewrite()">测试URL重写</button>
        <div id="urlRewriteResult"></div>
    </div>
    
    <div class="test-section">
        <h2>📄 静态资源测试</h2>
        <p>测试远程静态资源是否正确加载</p>
        <button onclick="testStaticResources()">测试静态资源</button>
        <div id="staticResourceResult"></div>
    </div>
    
    <div class="test-section">
        <h2>🔗 API代理测试</h2>
        <p>测试evernotecid://协议的API代理</p>
        <button onclick="testEvernotecidAPI()">测试API代理</button>
        <div id="apiProxyResult"></div>
    </div>
    
    <script>
        // 显示页面基本信息
        document.getElementById('currentUrl').textContent = window.location.href;
        document.getElementById('protocol').textContent = window.location.protocol;
        document.getElementById('hostname').textContent = window.location.hostname;
        document.getElementById('port').textContent = window.location.port;
        
        // 检查URL重写脚本是否已注入
        if (window.fetch !== window.fetch.__original) {
            console.log('✅ URL重写脚本已成功注入');
        } else {
            console.log('❌ URL重写脚本未注入或注入失败');
        }
        
        function testURLRewrite() {
            const resultDiv = document.getElementById('urlRewriteResult');
            resultDiv.innerHTML = '<p class="info">正在测试URL重写...</p>';
            
            try {
                // 测试URL重写功能
                const testURL = 'evernotecid://test-resource-id/sample/path';
                console.log('原始URL:', testURL);
                
                // 模拟fetch调用，观察URL是否被重写
                const originalFetch = window.fetch;
                window.fetch = function(resource, options) {
                    console.log('重写后URL:', resource);
                    resultDiv.innerHTML = `
                        <p class="success">✅ URL重写测试成功</p>
                        <p>原始URL: ${testURL}</p>
                        <p>重写后URL: ${resource}</p>
                    `;
                    // 恢复原始fetch
                    window.fetch = originalFetch;
                    // 返回一个rejected promise以避免实际请求
                    return Promise.reject('测试完成');
                };
                
                // 触发fetch调用
                fetch(testURL).catch(() => {});
                
            } catch (error) {
                resultDiv.innerHTML = `<p class="error">❌ URL重写测试失败: ${error}</p>`;
            }
        }
        
        function testStaticResources() {
            const resultDiv = document.getElementById('staticResourceResult');
            resultDiv.innerHTML = '<p class="info">正在测试静态资源...</p>';
            
            // 测试CSS文件加载
            const link = document.createElement('link');
            link.rel = 'stylesheet';
            link.href = '/test.css';
            link.onload = function() {
                console.log('✅ CSS文件加载成功');
                updateStaticResourceResult(true, 'CSS');
            };
            link.onerror = function() {
                console.log('❌ CSS文件加载失败');
                updateStaticResourceResult(false, 'CSS');
            };
            document.head.appendChild(link);
            
            // 测试JS文件加载
            const script = document.createElement('script');
            script.src = '/test.js';
            script.onload = function() {
                console.log('✅ JS文件加载成功');
                updateStaticResourceResult(true, 'JS');
            };
            script.onerror = function() {
                console.log('❌ JS文件加载失败');
                updateStaticResourceResult(false, 'JS');
            };
            document.head.appendChild(script);
        }
        
        function updateStaticResourceResult(success, type) {
            const resultDiv = document.getElementById('staticResourceResult');
            const statusClass = success ? 'success' : 'error';
            const statusIcon = success ? '✅' : '❌';
            resultDiv.innerHTML += `<p class="${statusClass}">${statusIcon} ${type}文件${success ? '加载成功' : '加载失败'}</p>`;
        }
        
        function testEvernotecidAPI() {
            const resultDiv = document.getElementById('apiProxyResult');
            resultDiv.innerHTML = '<p class="info">正在测试API代理...</p>';
            
            // 测试evernotecid协议请求
            const testAPIURL = 'evernotecid://test-resource-id/sample/resource';
            
            fetch(testAPIURL)
                .then(response => {
                    if (response.ok) {
                        resultDiv.innerHTML = `
                            <p class="success">✅ API代理测试成功</p>
                            <p>状态码: ${response.status}</p>
                            <p>Content-Type: ${response.headers.get('content-type')}</p>
                        `;
                    } else {
                        resultDiv.innerHTML = `
                            <p class="error">❌ API代理返回错误状态: ${response.status}</p>
                        `;
                    }
                })
                .catch(error => {
                    resultDiv.innerHTML = `
                        <p class="error">❌ API代理测试失败: ${error}</p>
                    `;
                });
        }
        
        // 页面加载完成后的日志
        console.log('🌐 远程HTML页面加载完成');
        console.log('📍 当前环境:', window.location.protocol === 'http:' ? 'HTTP代理环境' : '文件协议环境');
    </script>
</body>
</html>
```

#### test.css
```css
/* 测试CSS文件 */
body {
    background-color: #f0f8ff;
}

.test-section {
    background-color: #fff;
    border-radius: 5px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}
```

#### test.js
```javascript
// 测试JS文件
console.log('✅ 远程test.js文件加载成功');

window.remoteTestFunction = function() {
    console.log('远程JS函数执行成功');
    return true;
};
```

### 步骤2: 执行测试

#### 2.1 在应用中执行测试

```objc
// 在你的应用代码中
- (IBAction)testRemoteServer:(id)sender {
    ENWhiteboardEditorWKWebView *webView = self.whiteboardWebView;
    
    // 测试远程服务器
    [webView testRemoteServerConfiguration];
    
    // 等待几秒后检查状态
    dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(3.0 * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
        NSDictionary *status = [webView.httpServer serverStatus];
        AppLogInfo(@"📊 远程代理状态: %@", status);
        
        if ([status[@"mode"] isEqualToString:@"remote_proxy"]) {
            AppLogInfo(@"✅ 远程代理模式已启用");
            AppLogInfo(@"🌐 远程服务器: %@", status[@"remoteServerURL"]);
        } else {
            AppLogError(@"❌ 远程代理模式未启用");
        }
    });
}
```

#### 2.2 浏览器验证测试

获取本地代理URL后，也可以在外部浏览器中测试：

```objc
- (void)openInExternalBrowser {
    NSURL *proxyURL = [self.whiteboardWebView.httpServer indexHTMLURL];
    if (proxyURL) {
        [[NSWorkspace sharedWorkspace] openURL:proxyURL];
        AppLogInfo(@"🌐 在外部浏览器中打开: %@", proxyURL);
    }
}
```

### 步骤3: 验证结果

#### 3.1 日志验证
检查控制台输出，应该看到类似的日志：

```
[WhiteboardWebView] 🌐 配置远程服务器: http://10.228.32.31:8888
[WhiteboardWebView] 📋 HTML和静态资源将从远程服务器代理
[WhiteboardWebView] 🔄 evernotecid://协议仍在本地处理
[WhiteboardHTTP] 🌐 Proxying remote request: /index.html -> http://10.228.32.31:8888/index.html
[WhiteboardHTTP] Successfully proxied remote resource: http://10.228.32.31:8888/index.html (1234 bytes)
```

#### 3.2 功能验证清单

- [ ] **远程HTML加载**: 页面内容来自远程服务器
- [ ] **URL重写注入**: 脚本自动注入到远程HTML中
- [ ] **静态资源代理**: CSS/JS文件通过代理正确加载
- [ ] **evernotecid API**: 自定义协议请求正常工作
- [ ] **CORS支持**: 没有跨域错误
- [ ] **错误处理**: 网络错误有合适的错误处理

## 🎯 高级测试场景

### 场景1: 动态切换模式

```objc
- (void)testModeSwitching {
    ENWhiteboardEditorWKWebView *webView = self.whiteboardWebView;
    
    AppLogInfo(@"🔄 测试模式切换");
    
    // 1. 开始使用本地模式
    [webView configureRemoteServer:nil];
    [webView loadRequest:[NSURLRequest requestWithURL:[webView indexHTMLURL]]];
    
    // 2. 等待3秒后切换到远程模式
    dispatch_after(dispatch_time(DISPATCH_TIME_NOW, 3 * NSEC_PER_SEC), dispatch_get_main_queue(), ^{
        NSURL *remoteURL = [NSURL URLWithString:@"http://10.228.32.31:8888"];
        [webView configureRemoteServer:remoteURL];
        [webView loadRequest:[NSURLRequest requestWithURL:[webView indexHTMLURL]]];
        
        // 3. 再等待3秒后切换回本地模式
        dispatch_after(dispatch_time(DISPATCH_TIME_NOW, 3 * NSEC_PER_SEC), dispatch_get_main_queue(), ^{
            [webView configureRemoteServer:nil];
            [webView loadRequest:[NSURLRequest requestWithURL:[webView indexHTMLURL]]];
        });
    });
}
```

### 场景2: 多个远程服务器测试

```objc
- (void)testMultipleRemoteServers {
    NSArray *testServers = @[
        @"http://10.228.32.31:8888",
        @"http://192.168.1.100:3000",
        @"http://localhost:8080"
    ];
    
    ENWhiteboardEditorWKWebView *webView = self.whiteboardWebView;
    
    [testServers enumerateObjectsUsingBlock:^(NSString *serverURL, NSUInteger idx, BOOL *stop) {
        dispatch_after(dispatch_time(DISPATCH_TIME_NOW, idx * 5 * NSEC_PER_SEC), dispatch_get_main_queue(), ^{
            AppLogInfo(@"🧪 测试服务器 %lu: %@", (unsigned long)idx + 1, serverURL);
            [webView loadRemoteWhiteboardFromURL:serverURL];
        });
    }];
}
```

### 场景3: 错误处理测试

```objc
- (void)testErrorHandling {
    ENWhiteboardEditorWKWebView *webView = self.whiteboardWebView;
    
    // 测试无效URL
    [webView loadRemoteWhiteboardFromURL:@"invalid-url"];
    
    // 测试不存在的服务器
    [webView loadRemoteWhiteboardFromURL:@"http://non-existent-server.com"];
    
    // 测试连接超时
    [webView loadRemoteWhiteboardFromURL:@"http://10.255.255.1:9999"];
}
```

## 📊 性能测试

### 延迟测试

```objc
- (void)testProxyLatency {
    CFAbsoluteTime startTime = CFAbsoluteTimeGetCurrent();
    
    ENWhiteboardEditorWKWebView *webView = self.whiteboardWebView;
    [webView loadRemoteWhiteboardFromURL:@"http://10.228.32.31:8888"];
    
    // 监听页面加载完成
    [webView evaluateJavaScript:@"document.readyState" completionHandler:^(id result, NSError *error) {
        if ([result isEqualToString:@"complete"]) {
            CFAbsoluteTime endTime = CFAbsoluteTimeGetCurrent();
            CFTimeInterval latency = (endTime - startTime) * 1000; // 转换为毫秒
            AppLogInfo(@"📊 远程代理延迟: %.2f ms", latency);
        }
    }];
}
```

### 内存使用测试

```objc
- (void)testMemoryUsage {
    // 获取当前内存使用
    struct task_basic_info info;
    mach_msg_type_number_t size = sizeof(info);
    kern_return_t kerr = task_info(mach_task_self(), TASK_BASIC_INFO, (task_info_t)&info, &size);
    
    if (kerr == KERN_SUCCESS) {
        NSUInteger memoryBefore = info.resident_size / 1024 / 1024; // MB
        AppLogInfo(@"📊 代理前内存使用: %lu MB", (unsigned long)memoryBefore);
        
        // 启动远程代理
        ENWhiteboardEditorWKWebView *webView = self.whiteboardWebView;
        [webView loadRemoteWhiteboardFromURL:@"http://10.228.32.31:8888"];
        
        // 等待加载完成后再次检查
        dispatch_after(dispatch_time(DISPATCH_TIME_NOW, 5 * NSEC_PER_SEC), dispatch_get_main_queue(), ^{
            kern_return_t kerr2 = task_info(mach_task_self(), TASK_BASIC_INFO, (task_info_t)&info, &size);
            if (kerr2 == KERN_SUCCESS) {
                NSUInteger memoryAfter = info.resident_size / 1024 / 1024; // MB
                AppLogInfo(@"📊 代理后内存使用: %lu MB", (unsigned long)memoryAfter);
                AppLogInfo(@"📊 内存增加: %ld MB", (long)(memoryAfter - memoryBefore));
            }
        });
    }
}
```

这个测试指南提供了全面的远程HTML服务器测试方法，帮助验证所有功能是否正常工作。