# ENNavigationNarrowViewController 新增方法：getHiddenModuleItems

## 概述

为了优化 `showMoreItemMenu:` 方法中获取隐藏项目的逻辑，在 `ENNavigationNarrowViewController` 中新增了一个方法 `getHiddenModuleItems`，用于直接返回所有隐藏的模块项目数组。

## 新增方法

### 头文件 (ENNavigationNarrowViewController.h)

```objc
/**
 * 获取所有隐藏的模块项目，基于 ENNavigationManager 的 collapsedItemIdentifiers
 */
- (NSArray<ENNavigationStackViewModuleItem *> * _Nonnull)getHiddenModuleItems;
```

### 实现文件 (ENNavigationNarrowViewController.m)

```objc
- (NSArray<ENNavigationStackViewModuleItem *> *)getHiddenModuleItems {
  NSMutableArray<ENNavigationStackViewModuleItem *> *hiddenItems = [NSMutableArray array];
  
  // 获取所有可能隐藏的项目标识符
  NSArray<NSString *> *collapsedIdentifiers = [ENNavigationManager collapsedItemIdentifiers];
  
  // 遍历标识符，获取对应的模块项
  for (NSString *identifier in collapsedIdentifiers) {
    ENNavigationStackViewModuleItem *moduleItem = [self getViewModuleItemWithIdentifier:identifier];
    if (moduleItem) {
      [hiddenItems addObject:moduleItem];
    }
  }
  
  return [hiddenItems copy];
}
```

## 使用方式

### 原始的 showMoreItemMenu: 方法

```objc
- (void)showMoreItemMenu:(id)sender {
  self.moreWindowController = [[ENNavigationMoreWindowController alloc] initWithNoteCollectionWindowController:self];
  NSArray *hiddenItems = @[];
  if (self.navigationViewMode == ENNoteCollectionWindowControllerNavigationViewModeNarrow) {
    // 原始的复杂逻辑
    NSArray *hiddenViews = self.narrowNavigationViewController.stackView.detachedViews;
    hiddenItems = [hiddenViews en_mapWithBlock:^id(NSView *object) {
      return [self.narrowNavigationViewController stackViewItemForView:object];
    }];
  }
  NSView *moreItemView = [self viewWithIdentifier:kENNavigationOutlineViewControllerItemIdentifierMore makeVisible:YES subRect:nil];
  [self.moreWindowController showIn:moreItemView hiddenItems:hiddenItems];
  
  if (self.navigationViewMode == ENNoteCollectionWindowControllerNavigationViewModeNarrow) {
    [self.accountController.accountDecorator.analytics recordLegacyEventWithCategory:@"nav_bar_narrow" action:@"click" label:@"more"];
  } else {
    [self.accountController.accountDecorator.analytics recordLegacyEventWithCategory:@"nav_bar_wide" action:@"click" label:@"more"];
  }
}
```

### 优化后的 showMoreItemMenu: 方法

```objc
- (void)showMoreItemMenu:(id)sender {
  self.moreWindowController = [[ENNavigationMoreWindowController alloc] initWithNoteCollectionWindowController:self];
  NSArray *hiddenItems = @[];
  if (self.navigationViewMode == ENNoteCollectionWindowControllerNavigationViewModeNarrow) {
    // 使用新的简化方法
    hiddenItems = [self.narrowNavigationViewController getHiddenModuleItems];
  }
  NSView *moreItemView = [self viewWithIdentifier:kENNavigationOutlineViewControllerItemIdentifierMore makeVisible:YES subRect:nil];
  [self.moreWindowController showIn:moreItemView hiddenItems:hiddenItems];
  
  if (self.navigationViewMode == ENNoteCollectionWindowControllerNavigationViewModeNarrow) {
    [self.accountController.accountDecorator.analytics recordLegacyEventWithCategory:@"nav_bar_narrow" action:@"click" label:@"more"];
  } else {
    [self.accountController.accountDecorator.analytics recordLegacyEventWithCategory:@"nav_bar_wide" action:@"click" label:@"more"];
  }
}
```

## 方法设计说明

### 1. 数据源统一

- 使用 `ENNavigationManager.collapsedItemIdentifiers` 作为唯一的数据源
- 确保隐藏项目的定义在整个应用中保持一致

### 2. 现有方法复用

- 利用现有的 `getViewModuleItemWithIdentifier:` 方法获取模块项
- 保持与现有代码架构的一致性

### 3. 类型安全

- 返回强类型的 `NSArray<ENNavigationStackViewModuleItem *>` 
- 避免类型转换的复杂性和潜在错误

### 4. 性能优化

- 只创建真正需要的模块项
- 避免不必要的视图遍历操作

## collapsedItemIdentifiers 包含的标识符

根据 `ENNavigationManager.collapsedItemIdentifiers` 方法，当前包含以下模块：

```objc
[ENLibraryViewModule identifier]           // 资料库
[TemplateViewModule identifier]             // 模板
[TeamSpacePlaygroundViewModule identifier]  // 团队空间
[ENSharedWithMeModule identifier]           // 共享笔记
[NoteGraphViewModule identifier]            // 笔记图谱
[HomepageViewModule identifier]             // 主页
[LightNoteViewModule identifier]            // 便签
```

## 优势

1. **代码简化**：减少了复杂的视图遍历和类型转换逻辑
2. **维护性提高**：隐藏项目的逻辑集中在一个地方管理
3. **类型安全**：避免了运行时的类型转换错误
4. **性能优化**：只处理真正需要的项目，避免不必要的操作
5. **一致性**：确保整个应用中隐藏项目的定义保持一致

## 注意事项

1. 新方法依赖于 `ENNavigationManager.collapsedItemIdentifiers` 的正确性
2. 需要确保 `getViewModuleItemWithIdentifier:` 方法能正确处理所有标识符
3. 如果需要添加新的隐藏项目类型，只需要修改 `collapsedItemIdentifiers` 方法即可