# Evernote Mac Tag Drag & Drop 功能设计规范

## 1. 项目概述

### 1.1 功能描述
为 Evernote Mac 应用的标签模块新增拖拽功能，允许用户通过拖拽操作重新组织标签的层级结构，提升标签管理的用户体验。

### 1.2 设计目标
- 支持拖拽标签改变层级关系
- 保持现有排序逻辑的一致性
- 提供直观的视觉反馈
- 防止数据异常（循环引用）
- 复用现有的数据同步机制

## 2. 需求分析

### 2.1 功能范围

#### ✅ 支持的操作
- **层级重组**: 通过拖拽改变标签的父子关系
- **所有标签类型**: 本地标签、企业标签、共享标签均支持拖拽
- **无嵌套深度限制**: 支持任意深度的标签嵌套
- **同账户内操作**: 拖拽范围限制在当前账户内

#### ❌ 不支持的操作
- **同级排序**: 不支持调整同级标签的顺序（会被自动排序覆盖）
- **撤销功能**: 拖拽操作无撤销支持
- **跨账户拖拽**: 架构层面已限制在单账户内

### 2.2 用例场景

#### 场景1: 拖到节点上（改变层级）
```
初始状态:
A (parent: null)
├── A1 (parent: A)  
└── A2 (parent: A)

B (parent: null)
├── B1 (parent: B)
└── B2 (parent: B)

操作: 拖拽 B → A
结果: B.setParent(A) → B及其子标签成为A的子标签

最终状态:
A (parent: null)
├── A1 (parent: A)
├── A2 (parent: A)  
└── B (parent: A)
    ├── B1 (parent: B)
    └── B2 (parent: B)
```

#### 场景2: 拖到根级
```
操作: 拖拽 B1 → 空白区域
结果: B1.setParent(null) → B1成为顶级标签
```

### 2.3 技术约束

#### 现有架构限制
- **自动排序**: EN4TagTreeController 按 sortMethod 自动排序子标签
- **数据模型**: ENUniqueTag 通过 setParent() 管理层级关系
- **循环检查**: 已有循环引用保护机制

#### 拖拽API约束
```swift
func outlineView(_:validateDrop:proposedItem:proposedChildIndex:) -> NSDragOperation
// proposedChildIndex == -1: 拖到节点上
// proposedChildIndex >= 0: 拖到节点间（不支持）
```

## 3. 技术架构设计

### 3.1 模块职责

#### TagListOutlineView (UI层)
```swift
class TagListOutlineView: ENContentOutlineView {
    // 注册拖拽类型
    // 提供拖拽源支持
    // 处理拖拽目标验证
    // 执行拖拽操作
}
```

#### TagListDataSource (数据协调层)
```swift
extension TagListDataSource: NSOutlineViewDataSource {
    // 实现拖拽委托方法
    // 验证拖拽合法性
    // 调用数据模型更新
    // 处理UI刷新
}
```

#### ENUniqueTag (数据模型层)
```swift
// 现有方法，无需修改
func setParent(_:completion:)
// 自动处理:
// - 数据模型更新
// - 设置dirty标记
// - 发送通知
// - 触发同步
```

### 3.2 数据流设计

```
用户拖拽 → TagListOutlineView
    ↓
验证拖拽 → TagListDataSource.validateDrop()
    ↓
执行拖拽 → TagListDataSource.acceptDrop()
    ↓
更新模型 → ENUniqueTag.setParent()
    ↓
发送通知 → kENUniqueTagsHierarchyDidChangeNotification
    ↓
刷新UI → TagListDataSource.reloadOutlineDataWithAutoExpendItem()
    ↓
自动同步 → 现有同步机制
```

## 4. 实现方案

### 4.1 拖拽类型定义

```swift
// NSPasteboard 自定义类型
extension NSPasteboard.PasteboardType {
    static let tagDragType = NSPasteboard.PasteboardType("com.evernote.tag.drag")
}
```

### 4.2 TagListOutlineView 增强

```swift
extension TagListOutlineView {
    override func awakeFromNib() {
        super.awakeFromNib()
        registerForDraggedTypes([.tagDragType])
    }
}

extension TagListOutlineView: NSOutlineViewDataSource {
    // 拖拽源支持
    func outlineView(_ outlineView: NSOutlineView, 
                    pasteboardWriterForItem item: Any) -> NSPasteboardWriting? {
        guard let tag = item as? ENUniqueTag else { return nil }
        
        let pasteboardItem = NSPasteboardItem()
        pasteboardItem.setString(tag.localStoreIDAsString, 
                                forType: .tagDragType)
        return pasteboardItem
    }
    
    // 拖拽目标验证
    func outlineView(_ outlineView: NSOutlineView,
                    validateDrop info: NSDraggingInfo,
                    proposedItem item: Any?,
                    proposedChildIndex index: Int) -> NSDragOperation {
        
        // 只允许拖到节点上，不允许拖到节点间
        guard index == -1 else { return .none }
        
        guard let dataSource = dataSource as? TagListDataSource else { return .none }
        return dataSource.validateTagDrop(info: info, targetTag: item as? ENUniqueTag)
    }
    
    // 执行拖拽
    func outlineView(_ outlineView: NSOutlineView,
                    acceptDrop info: NSDraggingInfo,
                    item: Any?,
                    childIndex: Int) -> Bool {
        
        guard let dataSource = dataSource as? TagListDataSource else { return false }
        return dataSource.acceptTagDrop(info: info, newParent: item as? ENUniqueTag)
    }
}
```

### 4.3 TagListDataSource 拖拽逻辑

```swift
extension TagListDataSource {
    func validateTagDrop(info: NSDraggingInfo, targetTag: ENUniqueTag?) -> NSDragOperation {
        // 获取拖拽的标签
        guard let draggedTag = getDraggedTag(from: info) else { return .none }
        
        // 检查循环引用
        if wouldCreateCycle(dragging: draggedTag, to: targetTag) {
            return .none
        }
        
        // 检查是否是自己拖到自己
        if draggedTag == targetTag {
            return .none
        }
        
        return .move
    }
    
    func acceptTagDrop(info: NSDraggingInfo, newParent: ENUniqueTag?) -> Bool {
        guard let draggedTag = getDraggedTag(from: info) else { return false }
        
        draggedTag.setParent(newParent) { [weak self] _ in
            DispatchQueue.main.async {
                self?.reloadOutlineDataWithAutoExpendItem()
            }
        }
        
        return true
    }
    
    private func getDraggedTag(from info: NSDraggingInfo) -> ENUniqueTag? {
        guard let tagGUID = info.draggingPasteboard.string(forType: .tagDragType) else {
            return nil
        }
        
        return findItemByGUID(tagGUID)
    }
    
    private func wouldCreateCycle(dragging: ENUniqueTag, to target: ENUniqueTag?) -> Bool {
        guard let target = target else { return false }
        
        // 检查目标是否在拖拽标签的子树中
        return target.ancestors.contains(dragging)
    }
}
```

### 4.4 视觉反馈增强

```swift
extension TagListOutlineView {
    override func draggingEntered(_ sender: NSDraggingInfo) -> NSDragOperation {
        let operation = super.draggingEntered(sender)
        if operation != .none {
            // 添加拖拽进入效果
            addDragHighlight()
        }
        return operation
    }
    
    override func draggingExited(_ sender: NSDraggingInfo?) {
        super.draggingExited(sender)
        // 移除拖拽高亮效果
        removeDragHighlight()
    }
    
    private func addDragHighlight() {
        // 实现拖拽高亮效果
    }
    
    private func removeDragHighlight() {
        // 移除高亮效果
    }
}
```

## 5. 错误处理

### 5.1 拖拽验证失败
- **循环引用**: 显示禁止图标，不允许操作
- **自己拖到自己**: 显示禁止图标
- **拖到节点间**: 显示禁止图标

### 5.2 数据更新失败
- 依赖现有的 ENUniqueTag.setParent() 错误处理机制
- 同步失败时的恢复逻辑由现有架构处理

## 6. 性能考虑

### 6.1 拖拽响应性
- 拖拽验证逻辑需要高效执行
- 循环引用检查的时间复杂度: O(depth)

### 6.2 UI更新优化
- 使用现有的 reloadOutlineDataWithAutoExpendItem()
- 保持展开状态的恢复机制

## 7. 测试策略

### 7.1 功能测试
- 基本拖拽操作
- 循环引用防护
- 多层级嵌套
- 不同标签类型
- 拖拽到根级

### 7.2 边界测试
- 深层嵌套标签
- 大量子标签的标签
- 网络断开时的拖拽
- 同步冲突处理

### 7.3 用户体验测试
- 拖拽流畅性
- 视觉反馈及时性
- 错误状态提示

## 8. 兼容性说明

### 8.1 现有功能
- 不影响现有的标签创建、重命名、删除功能
- 保持现有的排序逻辑
- 兼容现有的搜索和筛选功能

### 8.2 数据格式
- 不改变 ENUniqueTag 数据模型
- 不影响现有的同步协议
- 保持与服务端的兼容性

## 9. 后续扩展

### 9.1 可能的增强
- 批量拖拽多个标签
- 支持键盘快捷键辅助拖拽
- 拖拽时显示层级预览

### 9.2 架构改进
- 如需支持自定义排序，可考虑添加 ENTagsSortOptionManual
- 优化大量标签时的性能表现

---

**文档版本**: v1.0  
**创建日期**: 2025-11-21  
**最后更新**: 2025-11-21  