# Evernote Mac Tag Drag & Drop 开发任务清单

## 📋 开发阶段规划

### Phase 1: 基础架构 (2-3天)
- [ ] **P1.1** 定义拖拽数据类型
- [ ] **P1.2** TagListOutlineView 注册拖拽支持  
- [ ] **P1.3** 实现基础拖拽源功能
- [ ] **P1.4** 实现基础拖拽目标功能
- [ ] **P1.5** 基础拖拽验证逻辑

### Phase 2: 核心功能 (3-4天) 
- [ ] **P2.1** 实现拖拽验证逻辑
- [ ] **P2.2** 循环引用检查机制
- [ ] **P2.3** 拖拽执行逻辑
- [ ] **P2.4** 数据模型更新集成
- [ ] **P2.5** UI刷新机制

### Phase 3: 用户体验 (2-3天)
- [ ] **P3.1** 拖拽视觉反馈
- [ ] **P3.2** 禁止操作的视觉提示
- [ ] **P3.3** 拖拽动画效果
- [ ] **P3.4** 错误状态处理
- [ ] **P3.5** 性能优化

### Phase 4: 测试验证 (2-3天)
- [ ] **P4.1** 单元测试
- [ ] **P4.2** 集成测试  
- [ ] **P4.3** 边界情况测试
- [ ] **P4.4** 用户体验测试
- [ ] **P4.5** 性能测试

---

## 🔧 详细任务分解

### Phase 1: 基础架构

#### P1.1 定义拖拽数据类型
**文件**: 新增 `TagDragDropConstants.swift`
```swift
// 创建拖拽数据类型定义
extension NSPasteboard.PasteboardType {
    static let tagDragType = NSPasteboard.PasteboardType("com.evernote.tag.drag")
}

// 拖拽数据模型
struct TagDragData {
    let tagGUID: String
    let tagName: String
}
```
**预估时间**: 0.5天  
**依赖**: 无

#### P1.2 TagListOutlineView 注册拖拽支持
**文件**: `TagListOutlineView.swift`
```swift
// 在 awakeFromNib 中注册拖拽类型
override func awakeFromNib() {
    super.awakeFromNib()
    registerForDraggedTypes([.tagDragType])
}
```
**预估时间**: 0.5天  
**依赖**: P1.1

#### P1.3 实现基础拖拽源功能
**文件**: `TagListOutlineView.swift`
```swift
// 实现 NSOutlineViewDataSource 拖拽源方法
func outlineView(_:pasteboardWriterForItem:) -> NSPasteboardWriting?
```
**预估时间**: 1天  
**依赖**: P1.1, P1.2

#### P1.4 实现基础拖拽目标功能  
**文件**: `TagListOutlineView.swift`
```swift
// 实现拖拽目标委托方法
func outlineView(_:validateDrop:proposedItem:proposedChildIndex:) -> NSDragOperation
func outlineView(_:acceptDrop:item:childIndex:) -> Bool
```
**预估时间**: 1天  
**依赖**: P1.3

#### P1.5 基础拖拽验证逻辑
**文件**: `TagListDataSource.swift`
```swift
// 实现基础验证方法
func validateTagDrop(info:targetTag:) -> NSDragOperation
private func getDraggedTag(from:) -> ENUniqueTag?
```
**预估时间**: 1天  
**依赖**: P1.4

### Phase 2: 核心功能

#### P2.1 实现拖拽验证逻辑
**文件**: `TagListDataSource.swift`  
**任务**:
- [ ] 验证 proposedChildIndex == -1
- [ ] 检查拖拽标签和目标标签的有效性
- [ ] 排除自己拖到自己的情况
- [ ] 返回正确的 NSDragOperation

**预估时间**: 1天  
**依赖**: P1.5

#### P2.2 循环引用检查机制
**文件**: `TagListDataSource.swift`
```swift
private func wouldCreateCycle(dragging: ENUniqueTag, to target: ENUniqueTag?) -> Bool {
    guard let target = target else { return false }
    return target.ancestors.contains(dragging)
}
```
**预估时间**: 1天  
**依赖**: P2.1

#### P2.3 拖拽执行逻辑  
**文件**: `TagListDataSource.swift`
```swift
func acceptTagDrop(info: NSDraggingInfo, newParent: ENUniqueTag?) -> Bool {
    // 调用 ENUniqueTag.setParent() 
    // 处理完成回调
}
```
**预估时间**: 1天  
**依赖**: P2.2

#### P2.4 数据模型更新集成
**文件**: `TagListDataSource.swift`
**任务**:
- [ ] 集成现有的 `ENUniqueTag.setParent()` 方法
- [ ] 处理异步更新回调
- [ ] 确保通知机制正常工作
- [ ] 验证同步逻辑

**预估时间**: 1天  
**依赖**: P2.3

#### P2.5 UI刷新机制
**文件**: `TagListDataSource.swift`
**任务**:
- [ ] 调用 `reloadOutlineDataWithAutoExpendItem()`
- [ ] 保持展开状态
- [ ] 处理选中状态
- [ ] 确保排序逻辑正确

**预估时间**: 1天  
**依赖**: P2.4

### Phase 3: 用户体验

#### P3.1 拖拽视觉反馈
**文件**: `TagListOutlineView.swift`
**任务**:
- [ ] 实现 `draggingEntered()` 高亮效果
- [ ] 实现 `draggingExited()` 移除高亮
- [ ] 添加拖拽时的视觉指示
- [ ] 优化拖拽图像显示

**预估时间**: 1天  
**依赖**: P2.5

#### P3.2 禁止操作的视觉提示
**文件**: `TagListOutlineView.swift`  
**任务**:
- [ ] 循环引用时显示禁止图标
- [ ] 无效拖拽时的视觉反馈
- [ ] 拖到节点间时的禁止提示
- [ ] 错误状态的用户提示

**预估时间**: 1天  
**依赖**: P3.1

#### P3.3 拖拽动画效果
**文件**: `TagListOutlineView.swift`
**任务**:
- [ ] 平滑的拖拽动画
- [ ] 释放时的回弹效果
- [ ] 成功操作的确认动画
- [ ] 性能优化

**预估时间**: 1天  
**依赖**: P3.2

#### P3.4 错误状态处理
**文件**: `TagListDataSource.swift`
**任务**:
- [ ] 网络异常时的处理
- [ ] 数据更新失败的恢复
- [ ] 用户友好的错误提示
- [ ] 日志记录和调试信息

**预估时间**: 0.5天  
**依赖**: P3.3

#### P3.5 性能优化
**文件**: `TagListDataSource.swift`, `TagListOutlineView.swift`
**任务**:
- [ ] 拖拽验证逻辑优化
- [ ] 减少不必要的UI更新
- [ ] 大量标签时的性能测试
- [ ] 内存使用优化

**预估时间**: 0.5天  
**依赖**: P3.4

### Phase 4: 测试验证

#### P4.1 单元测试
**文件**: 新增测试文件
**任务**:
- [ ] `wouldCreateCycle()` 方法测试
- [ ] `validateTagDrop()` 逻辑测试  
- [ ] `getDraggedTag()` 数据解析测试
- [ ] 边界条件测试

**预估时间**: 1天  
**依赖**: P3.5

#### P4.2 集成测试
**任务**:
- [ ] 拖拽到不同层级的集成测试
- [ ] 与现有排序逻辑的集成测试
- [ ] 与搜索功能的集成测试
- [ ] 数据同步的集成测试

**预估时间**: 1天  
**依赖**: P4.1

#### P4.3 边界情况测试
**任务**:
- [ ] 深层嵌套标签拖拽
- [ ] 大量子标签的性能测试
- [ ] 网络断开情况测试
- [ ] 并发操作测试

**预估时间**: 1天  
**依赖**: P4.2

#### P4.4 用户体验测试
**任务**:
- [ ] 拖拽流畅性测试
- [ ] 视觉反馈测试
- [ ] 错误提示测试
- [ ] 可用性测试

**预估时间**: 0.5天  
**依赖**: P4.3

#### P4.5 性能测试
**任务**:
- [ ] 拖拽响应时间测试
- [ ] 内存使用测试
- [ ] CPU使用率测试
- [ ] 大数据量场景测试

**预估时间**: 0.5天  
**依赖**: P4.4

---

## 🎯 里程碑检查点

### Milestone 1: 基础功能完成 (Phase 1 + Phase 2)
**目标日期**: 开发开始后第5-7天
**验收标准**:
- [ ] 能够拖拽标签到其他标签上
- [ ] 层级关系正确更新
- [ ] 循环引用被正确阻止
- [ ] UI正确刷新并保持排序

### Milestone 2: 用户体验完成 (Phase 3)
**目标日期**: 开发开始后第7-10天  
**验收标准**:
- [ ] 拖拽有明确的视觉反馈
- [ ] 禁止操作有清晰提示
- [ ] 拖拽动画流畅自然
- [ ] 错误状态处理完善

### Milestone 3: 测试验证完成 (Phase 4)
**目标日期**: 开发开始后第9-13天
**验收标准**:
- [ ] 所有测试用例通过
- [ ] 性能指标达标
- [ ] 边界情况处理正确
- [ ] 代码review完成

---

## 📝 开发注意事项

### 代码规范
- 遵循现有的Swift代码风格
- 添加必要的文档注释
- 使用现有的日志机制
- 保持与现有架构的一致性

### 安全考虑
- 验证所有用户输入
- 防止内存泄漏
- 确保线程安全
- 保护敏感数据

### 性能要求
- 拖拽响应时间 < 100ms
- 内存使用增长 < 5%
- 支持1000+标签的场景
- 60fps的流畅动画

### 兼容性要求
- 不破坏现有功能
- 保持数据格式兼容
- 支持现有的快捷键
- 适配不同屏幕尺寸

---

**总预估时间**: 9-13天  
**建议并行任务**: P4.1-P4.5可与P3后期并行进行  
**关键风险点**: 循环引用检查逻辑、UI刷新性能、数据同步一致性