# Evernote Mac AI Area 共享容器重构方案

## 问题背景

### 当前架构问题
1. **AI区域内容不一致**: 每个模块独立创建 `ENModuleAIContainerViewController`，切换模块时AI状态重置
2. **AI区域宽度不一致**: 不同模块的AI区域宽度独立保存，用户体验不连贯
3. **资源浪费**: 多个AI容器实例占用不必要的内存

### 问题根源代码
位置: `ENNoteCollectionWindowController+ViewModules.m:145-152`
```objc
// 🔥 问题代码：每个模块创建独立AI容器
if (!module.aiContainerViewController) {
    ENModuleAIContainerViewController *container = 
      [[ENModuleAIContainerViewController alloc] 
        initWithModule:module 
        accountController:self.accountController];
    module.aiContainerViewController = container;  // ❌ 每个模块独立容器
    [container restoreLayoutState];  // ❌ 每次恢复不同状态
}
```

## 重构目标

### 核心目标
- 所有支持AI的模块共享同一个AI容器实例
- AI聊天历史、状态、宽度在模块切换时保持一致
- 模块切换时只替换左侧内容，右侧AI区域保持不变

### 期望效果
```
用户场景：
1. 在Notes模块与AI聊天，调整AI区域宽度到350px
2. 切换到Tasks模块
3. AI聊天历史保持，AI区域仍然是350px宽度
4. 继续与AI对话，上下文连续
```

## 重构架构设计

### 架构对比图

**重构前（问题架构）:**
```
ENNoteCollectionWindowController
├── EN4NotesModule 
│   └── aiContainerViewController(实例1) ❌
├── TaskViewModule
│   └── aiContainerViewController(实例2) ❌
└── LibraryViewModule
    └── aiContainerViewController(实例3) ❌
```

**重构后（目标架构）:**
```
ENNoteCollectionWindowController
├── sharedAIContainer: ENModuleAIContainerViewController ✅
│   ├── moduleContainerView (动态切换: Notes/Tasks/Library)
│   └── aiContainerView (保持不变: AICopilotViewController)
├── EN4NotesModule (纯净模块) ✅
├── TaskViewModule (纯净模块) ✅
└── LibraryViewModule (纯净模块) ✅
```

### 详细架构图

```
ENNoteCollectionWindowController
└── setModule: 方法重构
    ├── 支持AI的模块 ([module.class supportsAIArea])
    │   ├── 创建/获取 sharedAIContainer
    │   ├── [sharedAIContainer switchToModule:module]
    │   └── 显示 sharedAIContainer.view
    └── 不支持AI的模块
        ├── 隐藏 sharedAIContainer (如果显示中)
        └── 显示 module.view

ENModuleAIContainerViewController (重构)
├── moduleContainerView
│   └── currentModule (动态切换的模块)
├── aiContainerView  
│   └── aiViewController (共享的AI实例)
└── switchToModule:animated: (新增方法)
```

## 具体实现方案

### 1. 窗口控制器层面修改

#### 1.1 ENNoteCollectionWindowController+ViewModules.h
```objc
@interface ENNoteCollectionWindowController (ViewModules)

// 🆕 新增：共享AI容器属性
@property (strong, readonly) ENModuleAIContainerViewController *sharedAIContainer;

// 🆕 新增：AI容器管理方法
- (void)createSharedAIContainerIfNeeded;
- (void)hideSharedAIContainerIfNeeded;

// 现有属性保持不变...
@property (weak, readonly) ENBaseViewModule *currentModule;
@property (weak, readonly) ENBaseViewModule *previousModule;

@end
```

#### 1.2 ENNoteCollectionWindowControllerPrivate.h
```objc
// 添加私有属性
@interface ENNoteCollectionWindowController ()
@property (strong) ENModuleAIContainerViewController *sharedAIContainer;
@end
```

#### 1.3 setModule: 方法重构
```objc
- (void)setModule:(ENBaseViewModule *)module {
    // ... 前面的隐藏逻辑保持不变 ...
    
    // Show the new one
    if (module) {
        NSView *viewToAdd;
        NSViewController *controllerToAdd;
        
        // 🔥 核心重构：检查模块AI支持，使用共享AI容器
        if ([module.class supportsAIArea]) {
            // 支持AI的模块：使用共享AI容器
            [self createSharedAIContainerIfNeeded];
            
            // 切换容器内的模块，而不是创建新容器
            [self.sharedAIContainer switchToModule:module animated:YES];
            
            viewToAdd = self.sharedAIContainer.view;
            controllerToAdd = self.sharedAIContainer;
            
        } else {
            // 不支持AI的模块：直接显示模块，隐藏AI容器
            [self hideSharedAIContainerIfNeeded];
            
            viewToAdd = module.view;
            controllerToAdd = module;
        }
        
        // ... 后面的显示逻辑保持不变 ...
    }
}

- (void)createSharedAIContainerIfNeeded {
    if (!_sharedAIContainer) {
        _sharedAIContainer = [[ENModuleAIContainerViewController alloc] 
                               initWithAccountController:self.accountController];
        
        // 只在首次创建时恢复布局状态
        [_sharedAIContainer restoreLayoutState];
    }
}

- (void)hideSharedAIContainerIfNeeded {
    if (_sharedAIContainer && _sharedAIContainer.parentViewController) {
        // 隐藏但不销毁共享容器，保持状态
        [_sharedAIContainer.view removeFromSuperview];
        // 注意：不设置 _sharedAIContainer = nil，保持实例
    }
}
```

### 2. AI容器重构

#### 2.1 ENModuleAIContainerViewController.h 重构
```objc
@interface ENModuleAIContainerViewController : NSViewController <NSSplitViewDelegate>

// 🔄 修改：从包装单个模块改为动态切换模块
@property (weak) ENBaseViewModule *currentModule;  // 当前显示的模块
@property (strong, readonly) NSViewController *aiViewController; // 共享AI实例

// XIB连接保持不变
@property (weak) IBOutlet NSSplitView *splitView;
@property (weak) IBOutlet NSView *moduleContainerView;  // 重命名：从wrappedModuleView
@property (weak) IBOutlet NSView *aiContainerView;

// 🆕 新增：模块切换方法
- (void)switchToModule:(ENBaseViewModule *)module animated:(BOOL)animated;

// 🔄 修改：构造方法不再需要包装特定模块
- (instancetype)initWithAccountController:(ENAccountController *)accountController;

// 状态管理
- (void)saveLayoutState;
- (void)restoreLayoutState;

@end
```

#### 2.2 模块切换实现
```objc
@implementation ENModuleAIContainerViewController

- (void)switchToModule:(ENBaseViewModule *)module animated:(BOOL)animated {
    if (self.currentModule == module) {
        return;
    }
    
    ENBaseViewModule *previousModule = self.currentModule;
    
    // 隐藏前一个模块
    if (previousModule) {
        [previousModule windowControllerWillHideContentView];
        [previousModule.view removeFromSuperview];
        [previousModule windowControllerDidHideContentView];
    }
    
    // 显示新模块
    if (module) {
        [module windowControllerWillShowContentView];
        
        // 添加模块视图到容器
        module.view.translatesAutoresizingMaskIntoConstraints = NO;
        [self.moduleContainerView addSubview:module.view];
        
        // 设置约束
        [NSLayoutConstraint activateConstraints:@[
            [module.view.topAnchor constraintEqualToAnchor:self.moduleContainerView.topAnchor],
            [module.view.leadingAnchor constraintEqualToAnchor:self.moduleContainerView.leadingAnchor],
            [module.view.trailingAnchor constraintEqualToAnchor:self.moduleContainerView.trailingAnchor],
            [module.view.bottomAnchor constraintEqualToAnchor:self.moduleContainerView.bottomAnchor]
        ]];
        
        if (animated) {
            // 淡入动画
            module.view.alphaValue = 0.0;
            [NSAnimationContext runAnimationGroup:^(NSAnimationContext *context) {
                context.duration = 0.25;
                module.view.animator.alphaValue = 1.0;
            } completionHandler:^{
                [module windowControllerDidShowContentView];
            }];
        } else {
            [module windowControllerDidShowContentView];
        }
    }
    
    _currentModule = module;
}

// 🔄 修改：消息转发到当前模块
- (void)forwardInvocation:(NSInvocation *)invocation {
    if (self.currentModule && [self.currentModule respondsToSelector:invocation.selector]) {
        [invocation invokeWithTarget:self.currentModule];
    } else {
        [super forwardInvocation:invocation];
    }
}

- (NSMethodSignature *)methodSignatureForSelector:(SEL)selector {
    NSMethodSignature *signature = [super methodSignatureForSelector:selector];
    if (!signature && self.currentModule) {
        signature = [self.currentModule methodSignatureForSelector:selector];
    }
    return signature;
}

- (BOOL)respondsToSelector:(SEL)selector {
    return [super respondsToSelector:selector] || 
           (self.currentModule && [self.currentModule respondsToSelector:selector]);
}

@end
```

### 3. 清理模块代码

#### 3.1 ENBaseViewModule.h 部分简化
```objc
@interface ENBaseViewModule : NSObject

// 🗑️ 移除：不再需要每个模块持有AI容器
// @property (strong) ENModuleAIContainerViewController *aiContainerViewController;  // 删除

// ✅ 保留：为未来不支持AI的模块预留灵活性
+ (BOOL)supportsAIArea;           // 保留 - 默认返回YES，个别模块可重写为NO

// 🗑️ 移除：全局统一管理，不需要模块级配置
// + (BOOL)shouldShowAIAreaByDefault; // 删除 - 全局默认隐藏
// + (CGFloat)defaultAIAreaWidth;     // 删除 - 全局默认320px

// 🗑️ 移除：实例方法改为在窗口控制器层面管理
// - (void)showAIArea:(BOOL)show;  // 删除
// - (void)toggleAIArea;           // 删除

@end
```

#### 3.2 模块配置大部分移除，保留灵活性
```objc
// ENBaseViewModule.m - 提供默认实现
+ (BOOL)supportsAIArea { 
    return YES;  // 默认所有模块都支持AI
}

// ✅ EN4NotesModule.m - 保持默认，无需重写
// 使用基类默认实现 supportsAIArea = YES

// 🗑️ 移除不需要的配置方法
// + (BOOL)shouldShowAIAreaByDefault { return YES; }   // 删除
// + (CGFloat)defaultAIAreaWidth { return 300.0f; }    // 删除

// ✅ TaskViewModule.swift - 保持默认，无需重写
// 使用基类默认实现 supportsAIArea = YES

// 🗑️ 移除不需要的配置方法
// override class func shouldShowAIAreaByDefault() -> Bool { return true }   // 删除
// override class func defaultAIAreaWidth() -> CGFloat { return 280.0 }      // 删除

// ✅ LibraryViewModule.swift - 保持默认，无需重写
// 使用基类默认实现 supportsAIArea = YES

// 🗑️ 移除不需要的配置方法
// override class func shouldShowAIAreaByDefault() -> Bool { return false }  // 删除
// override class func defaultAIAreaWidth() -> CGFloat { return 320.0 }      // 删除

// 💡 未来如果某个模块不支持AI，可以这样重写：
// + (BOOL)supportsAIArea { return NO; }
```

#### 3.3 全局AI配置
```objc
// 在ENModuleAIContainerViewController中定义全局默认值
#define EN_AI_AREA_DEFAULT_WIDTH        320.0f   // 全局默认宽度
#define EN_AI_AREA_DEFAULT_VISIBLE      NO       // 全局默认隐藏

// 所有模块都支持AI，无需检查 supportsAIArea
```

### 4. 状态管理优化

#### 4.1 全局AI状态管理
```objc
// 全局偏好设置键
extern NSString * const ENGlobalAIAreaVisibleKey;    // "ENGlobalAIAreaVisible"
extern NSString * const ENGlobalAIAreaWidthKey;      // "ENGlobalAIAreaWidth"

// 🗑️ 移除所有基于moduleID的偏好设置键
// #define ENAIAreaVisibleKey(moduleID) [NSString stringWithFormat:@"EN%@AIAreaVisible", moduleID]  // 删除
// #define ENAIAreaWidthKey(moduleID) [NSString stringWithFormat:@"EN%@AIAreaWidth", moduleID]      // 删除

@implementation ENModuleAIContainerViewController

- (void)saveLayoutState {
    // 保存全局AI区域状态（不再基于模块ID）
    [[NSUserDefaults standardUserDefaults] setBool:self.isAIAreaVisible 
                                             forKey:ENGlobalAIAreaVisibleKey];
    [[NSUserDefaults standardUserDefaults] setFloat:self.aiAreaWidth 
                                              forKey:ENGlobalAIAreaWidthKey];
}

- (void)restoreLayoutState {
    // 恢复全局AI区域状态
    BOOL visible = [[NSUserDefaults standardUserDefaults] boolForKey:ENGlobalAIAreaVisibleKey];
    CGFloat width = [[NSUserDefaults standardUserDefaults] floatForKey:ENGlobalAIAreaWidthKey];
    
    if (width <= 0) {
        // 首次使用，使用全局默认宽度
        width = EN_AI_AREA_DEFAULT_WIDTH;  // 320px
    }
    
    [self setAIAreaVisible:visible width:width animated:NO];
}

@end
```

## 兼容性处理

### 现有代码兼容性
1. **currentModule访问**: 继续返回原始模块，不影响现有代码
2. **生命周期方法**: 通过容器正确转发到当前模块
3. **响应链**: 通过消息转发维持正确的响应链

### 风险控制
1. **渐进式重构**: 先实现核心逻辑，再逐步清理
2. **向后兼容**: 确保现有功能不受影响
3. **测试覆盖**: 重点测试模块切换和AI功能

## 实现优势

### 用户体验提升
- ✅ AI聊天历史在模块间保持
- ✅ AI区域宽度全局一致
- ✅ 切换模块时AI状态连续

### 技术优势  
- ✅ 内存使用优化（单一AI容器实例）
- ✅ 代码简化（移除模块级AI容器管理）
- ✅ 状态管理集中化

### 维护性提升
- ✅ AI相关逻辑集中管理
- ✅ 减少代码重复
- ✅ 更清晰的架构层次

## 后续扩展性

这种设计为未来扩展提供了良好基础：
- 支持AI功能的个性化配置
- 支持AI区域的布局模式切换
- 支持更复杂的AI交互场景