# YX-182 番茄钟状态栏功能 - API接口说明

## 1. 主要接口概览

### ENPomodoroStatusManager 核心接口

```objc
/**
 * 处理来自网页的状态变更通知
 * @param info 包含番茄钟状态信息的字典
 */
- (void)handleStateChangeNotification:(NSDictionary *)info;

/**
 * 显示状态栏图标
 */
- (void)showStatusItem;

/**
 * 隐藏状态栏图标
 */
- (void)hideStatusItem;
```

## 2. 通知数据格式

### 输入数据结构 (来自 PomodoroViewController)

```swift
// Swift 端发送的数据格式
let notificationInfo: [String: Any] = [
    "startTime": 1640995200.0,      // Unix时间戳 (NSTimeInterval)
    "type": "work",                 // "work" 或 "break" 
    "duration": 1500.0,             // 持续时长，秒 (NSTimeInterval)
    "state": "started"              // "started", "paused", "stopped", "resumed"
]
```

### 数据字段说明

| 字段名 | 类型 | 说明 | 可能值 |
|--------|------|------|--------|
| `startTime` | NSTimeInterval | Unix时间戳，番茄钟开始的绝对时间 | 1640995200.0 |
| `type` | NSString | 番茄钟类型 | "work", "break" |
| `duration` | NSTimeInterval | 持续时长（秒） | 1500.0 (25分钟), 300.0 (5分钟) |
| `state` | NSString | 当前操作状态 | "started", "paused", "stopped", "resumed" |

### 数据验证规则

```objc
// 数据验证示例
- (BOOL)validateNotificationInfo:(NSDictionary *)info {
    // 检查必要字段
    if (!info[@"startTime"] || !info[@"type"] || 
        !info[@"duration"] || !info[@"state"]) {
        return NO;
    }
    
    // 验证时间戳
    NSTimeInterval startTime = [info[@"startTime"] doubleValue];
    if (startTime <= 0) {
        return NO;
    }
    
    // 验证持续时长
    NSTimeInterval duration = [info[@"duration"] doubleValue];  
    if (duration <= 0 || duration > 7200) { // 最大2小时
        return NO;
    }
    
    // 验证类型
    NSString *type = info[@"type"];
    if (![type isEqualToString:@"work"] && ![type isEqualToString:@"break"]) {
        return NO;
    }
    
    // 验证状态
    NSString *state = info[@"state"];
    NSArray *validStates = @[@"started", @"paused", @"stopped", @"resumed"];
    if (![validStates containsObject:state]) {
        return NO;
    }
    
    return YES;
}
```

## 3. 状态转换逻辑

### 状态机定义

```objc
typedef NS_ENUM(NSInteger, ENPomodoroManagerState) {
    ENPomodoroManagerStateIdle,      // 空闲状态
    ENPomodoroManagerStateActive,    // 活跃运行
    ENPomodoroManagerStatePaused,    // 暂停状态
};
```

### 状态转换表

| 当前状态 | 接收通知 | 目标状态 | 操作 |
|----------|----------|----------|------|
| Idle | started | Active | 显示状态栏，开始倒计时 |
| Active | paused | Paused | 暂停倒计时，界面变灰 |
| Paused | resumed | Active | 恢复倒计时，恢复颜色 |
| Active/Paused | stopped | Idle | 隐藏状态栏，清理状态 |
| Active | started | Active | 重新开始新的番茄钟 |

### 状态处理方法

```objc
- (void)handleStateChangeNotification:(NSDictionary *)info {
    if (![self validateNotificationInfo:info]) {
        AppLogError(@"Invalid pomodoro notification data: %@", info);
        return;
    }
    
    NSString *state = info[@"state"];
    
    if ([state isEqualToString:@"started"]) {
        [self startPomodoroWithInfo:info];
    } else if ([state isEqualToString:@"paused"]) {
        [self pausePomodoro];
    } else if ([state isEqualToString:@"resumed"]) {
        [self resumePomodoro];  
    } else if ([state isEqualToString:@"stopped"]) {
        [self stopPomodoro];
    }
}
```

## 4. 时间计算逻辑

### 剩余时间计算

```objc
/**
 * 计算剩余时间
 * @param startTime 开始时间戳
 * @param duration 总持续时长
 * @param pausedDuration 已暂停的总时长
 * @return 剩余时间（秒）
 */
- (NSTimeInterval)calculateRemainingTimeWithStartTime:(NSTimeInterval)startTime
                                             duration:(NSTimeInterval)duration
                                       pausedDuration:(NSTimeInterval)pausedDuration {
    NSTimeInterval now = [[NSDate date] timeIntervalSince1970];
    NSTimeInterval elapsed = now - startTime - pausedDuration;
    NSTimeInterval remaining = duration - elapsed;
    
    return MAX(0, remaining);
}
```

### 进度计算

```objc
/**
 * 计算进度百分比
 * @param remainingTime 剩余时间
 * @param totalTime 总时间
 * @return 进度百分比 (0.0-1.0)
 */
- (CGFloat)calculateProgress:(NSTimeInterval)remainingTime totalTime:(NSTimeInterval)totalTime {
    if (totalTime <= 0) return 0.0;
    return (totalTime - remainingTime) / totalTime;
}
```

## 5. 用户交互接口

### 右键菜单回调

```objc
/**
 * 打开主窗口并跳转到番茄钟页面
 */
- (void)openMainWindow:(id)sender {
    ENNoteCollectionWindowController *mainWindow = 
        [self.accountController.mainNoteCollectionWindowController];
    
    if (mainWindow) {
        [mainWindow.window makeKeyAndOrderFront:nil];
        [NSApp activateIgnoringOtherApps:YES];
        [mainWindow showPomodoro];
    }
}

/**
 * 退出番茄钟模式
 */
- (void)exitPomodoro:(id)sender {
    [self hideStatusItem];
    [self stopPomodoro];
}
```

### 视图更新接口

```objc
// ENPomodoroStatusView 公开接口
- (void)updateDisplayWithRemainingTime:(NSTimeInterval)remainingTime
                             totalTime:(NSTimeInterval)totalTime
                           isBreakTime:(BOOL)isBreakTime
                              isPaused:(BOOL)isPaused;
```

## 6. 集成接口

### ENAccountController 集成

```objc
// ENAccountController.h 新增属性
@property (strong, nonnull, readonly) ENPomodoroStatusManager *pomodoroStatusManager;

// ENAccountController.m 初始化代码
- (instancetype)initWithAccountDecorator:(ENMacEvernoteAccountDecorator *)accountDecorator {
    // ... 现有代码
    _pomodoroStatusManager = [[ENPomodoroStatusManager alloc] 
                             initWithAccountController:self];
    // ... 
}
```

### PomodoroViewController 集成

```swift
// PomodoroViewController.swift 修改
func handleNotificationStateChange(_ info: [String: Any]) {
    AppLogInfo("[Pomodoro] State change: \(info)")
    
    // 获取主账户控制器
    guard let mainAccountController = ENAccountControllersManager.shared().mainAccountController else {
        AppLogError("[Pomodoro] No main account controller available")
        return
    }
    
    // 调用状态管理器处理通知
    mainAccountController.pomodoroStatusManager.handleStateChangeNotification(info)
}
```

## 7. 错误处理和日志

### 日志规范

```objc
// 使用统一的日志前缀
#define POMODORO_LOG_PREFIX @"[PomodoroStatus]"

// 不同级别的日志示例
AppLogInfo(@"%@ Pomodoro started: %@", POMODORO_LOG_PREFIX, info);
AppLogError(@"%@ Invalid notification data: %@", POMODORO_LOG_PREFIX, info);
AppLogDebug(@"%@ Timer updated, remaining: %.0f seconds", POMODORO_LOG_PREFIX, remainingTime);
```

### 错误代码定义

```objc
typedef NS_ENUM(NSInteger, ENPomodoroError) {
    ENPomodoroErrorNone = 0,
    ENPomodoroErrorInvalidData = 1001,
    ENPomodoroErrorInvalidState = 1002,
    ENPomodoroErrorTimerFailure = 1003,
    ENPomodoroErrorAccountNotAvailable = 1004,
};
```

## 8. 性能和资源管理

### Timer 管理

```objc
// 高效的定时器管理
- (void)startUpdateTimer {
    [self stopUpdateTimer]; // 先停止现有的
    
    self.updateTimer = [NSTimer scheduledTimerWithTimeInterval:1.0
                                                       target:self
                                                     selector:@selector(timerTick:)
                                                     userInfo:nil
                                                      repeats:YES];
    
    // 确保在 RunLoop 中正确运行
    [[NSRunLoop currentRunLoop] addTimer:self.updateTimer 
                                 forMode:NSRunLoopCommonModes];
}

- (void)stopUpdateTimer {
    [self.updateTimer invalidate];
    self.updateTimer = nil;
}
```

### 内存管理

```objc
- (void)dealloc {
    [self stopUpdateTimer];
    [self hideStatusItem];
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}
```

这个API接口说明确保了：
- 清晰的数据格式定义
- 完整的状态转换逻辑
- 详细的集成指南
- 统一的错误处理机制
- 高效的资源管理