# ENPomodoroStatusManager Implementation Plan

## Overview

This document outlines the implementation plan for updating ENPomodoroStatusManager to support the new state management system.

## Code Changes Required

### 1. Update Data Structures

#### Current State Enum (to be updated)
```objc
typedef NS_ENUM(NSInteger, ENPomodoroManagerState) {
    ENPomodoroManagerStateIdle,      // Remove
    ENPomodoroManagerStateActive,    // Remove  
    ENPomodoroManagerStatePaused,    // Remove
};
```

#### New State Enum
```objc
typedef NS_ENUM(NSInteger, ENPomodoroRunState) {
    ENPomodoroRunStateUnStarted = 1,      // 未开始/结束
    ENPomodoroRunStateFocusing = 2,       // 专注中
    ENPomodoroRunStateFocusPaused = 3,    // 专注暂停
    ENPomodoroRunStateFocusCompleted = 4, // 专注完成
    ENPomodoroRunStateBreaking = 5,       // 休息中
    ENPomodoroRunStateBreakCompleted = 6  // 休息完成
};
```

#### Update Private Properties
```objc
@interface ENPomodoroStatusManager ()
// Replace old properties
@property (assign, nonatomic) ENPomodoroRunState runState;
@property (assign, nonatomic) NSTimeInterval startTime;
@property (assign, nonatomic) NSTimeInterval remaining;
// Remove: pausedDuration, pauseStartTime, isBreakTime
@end
```

### 2. Update Notification Handling

#### New handleStateChangeNotification Method
```objc
- (void)handleStateChangeNotification:(NSDictionary *)info {
    // Validate new data structure
    if (![self validateNotificationInfo:info]) {
        return;
    }
    
    // Parse new fields
    NSInteger runState = [info[@"runState"] integerValue];
    self.startTime = [info[@"startTime"] doubleValue];
    self.remaining = [info[@"remaining"] doubleValue];
    // pauseTime ignored for now
    
    // Handle state changes
    [self handleRunStateChange:(ENPomodoroRunState)runState];
}
```

#### New State Handler Method
```objc
- (void)handleRunStateChange:(ENPomodoroRunState)newState {
    self.runState = newState;
    
    switch (newState) {
        case ENPomodoroRunStateUnStarted:
            [self handleUnStarted];
            break;
        case ENPomodoroRunStateFocusing:
            [self handleFocusing];
            break;
        case ENPomodoroRunStateFocusPaused:
            [self handleFocusPaused];
            break;
        case ENPomodoroRunStateFocusCompleted:
            [self handleFocusCompleted];
            break;
        case ENPomodoroRunStateBreaking:
            [self handleBreaking];
            break;
        case ENPomodoroRunStateBreakCompleted:
            [self handleBreakCompleted];
            break;
    }
}
```

### 3. Update Display Logic

#### New Time Calculation
```objc
- (NSTimeInterval)calculateActualRemainingTime {
    if (self.runState == ENPomodoroRunStateFocusPaused) {
        // Return frozen remaining time for paused state
        return self.remaining;
    }
    
    if (self.runState == ENPomodoroRunStateFocusing || 
        self.runState == ENPomodoroRunStateBreaking) {
        // Calculate real-time remaining
        NSTimeInterval now = [[NSDate date] timeIntervalSince1970];
        NSTimeInterval elapsed = now - self.startTime;
        return MAX(0, self.remaining - elapsed);
    }
    
    // For completed states
    return 0;
}
```

#### New Progress Calculation
```objc
- (CGFloat)calculateProgress {
    if (self.runState == ENPomodoroRunStateFocusCompleted ||
        self.runState == ENPomodoroRunStateBreakCompleted) {
        return 1.0; // Full progress for completed states
    }
    
    NSTimeInterval actualRemaining = [self calculateActualRemainingTime];
    if (self.remaining <= 0) return 0.0;
    
    NSTimeInterval elapsed = self.remaining - actualRemaining;
    return elapsed / self.remaining;
}
```

#### Update Icon Generation
```objc
- (NSImage *)generateIconWithProgress:(CGFloat)progress {
    CGFloat iconSize = 18.0;
    
    switch (self.runState) {
        case ENPomodoroRunStateFocusing:
            return [ENPomodoroIconRenderer workImageWithProgress:progress size:iconSize];
        case ENPomodoroRunStateFocusPaused:
            return [ENPomodoroIconRenderer pausedImageWithProgress:progress size:iconSize];
        case ENPomodoroRunStateFocusCompleted:
            return [ENPomodoroIconRenderer workImageWithProgress:1.0 size:iconSize];
        case ENPomodoroRunStateBreaking:
            return [ENPomodoroIconRenderer breakImageWithProgress:progress size:iconSize];
        case ENPomodoroRunStateBreakCompleted:
            return [ENPomodoroIconRenderer breakImageWithProgress:1.0 size:iconSize];
        case ENPomodoroRunStateUnStarted:
        default:
            return nil; // Will hide status item
    }
}
```

### 4. Update Text Display

#### New Text Generation
```objc
- (NSString *)generateDisplayText {
    switch (self.runState) {
        case ENPomodoroRunStateFocusing:
        case ENPomodoroRunStateFocusPaused:
        case ENPomodoroRunStateBreaking: {
            NSTimeInterval remaining = [self calculateActualRemainingTime];
            return [self formatTime:remaining];
        }
        case ENPomodoroRunStateFocusCompleted:
            return @"\u2003专注完成";
        case ENPomodoroRunStateBreakCompleted:
            return @"\u2003休息完成";
        case ENPomodoroRunStateUnStarted:
        default:
            return @"";
    }
}
```

### 5. Update State Handlers

```objc
- (void)handleUnStarted {
    [self hideStatusItem];
}

- (void)handleFocusing {
    [self showStatusItem];
    [self startUpdateTimer];
}

- (void)handleFocusPaused {
    [self stopUpdateTimer]; // Freeze updates
}

- (void)handleFocusCompleted {
    [self stopUpdateTimer];
    [self showCompletionNotification:@"专注完成" 
                             message:@"You've finished 1 Pomodoro. Take a break."];
    [self updateDisplay]; // Show completion state
}

- (void)handleBreaking {
    [self showStatusItem];
    [self startUpdateTimer];
}

- (void)handleBreakCompleted {
    [self stopUpdateTimer];
    [self showCompletionNotification:@"休息完成"
                             message:@"Break time is over. Ready to work!"];
    [self updateDisplay]; // Show completion state
}
```

### 6. Update Validation

#### New Validation Method
```objc
- (BOOL)validateNotificationInfo:(NSDictionary *)info {
    // Check required fields
    if (!info[@"runState"] || !info[@"startTime"] || !info[@"remaining"]) {
        return NO;
    }
    
    // Validate runState range
    NSInteger runState = [info[@"runState"] integerValue];
    if (runState < 1 || runState > 6) {
        return NO;
    }
    
    // Validate timestamps
    NSTimeInterval startTime = [info[@"startTime"] doubleValue];
    if (startTime <= 0) {
        return NO;
    }
    
    // Validate remaining time
    NSTimeInterval remaining = [info[@"remaining"] doubleValue];
    if (remaining < 0 || remaining > 7200) { // Max 2 hours
        return NO;
    }
    
    return YES;
}
```

## Implementation Steps

1. **Backup Current Implementation**: Create backup of current working code
2. **Update Enums and Properties**: Replace old state management with new system  
3. **Update Notification Handler**: Implement new data structure parsing
4. **Update Display Logic**: Implement new time calculation and progress logic
5. **Update Icon and Text Generation**: Support new states and completion messages
6. **Update State Handlers**: Implement handlers for each new state
7. **Update Validation**: Implement validation for new data structure
8. **Testing**: Comprehensive testing with all state transitions
9. **Cleanup**: Remove deprecated methods and properties

## Testing Strategy

- Test each state transition individually
- Verify time calculations are accurate
- Confirm UI updates correctly for each state
- Test completion notifications
- Verify status item hide/show behavior
- Test pause/resume functionality
- Validate edge cases (negative remaining, invalid states)

## Backward Compatibility

This is a breaking change that requires web component updates. The old and new systems cannot coexist. Migration should be coordinated between web and native components.