# Pomodoro State Management Specification

## Overview

This document defines the new state management system for the ENPomodoroStatusManager, based on the updated notification data structure from the web component.

## Data Structure

### Notification Fields

The web component now sends notifications with the following structure:

```json
{
  "runState": "focusing",       // State string
  "startTime": 1234567890123,   // Unix timestamp in milliseconds (start time) - can be null for "unStarted"
  "focusDuration": 1500,        // Total focus session duration in seconds
  "restDuration": 300,          // Total rest session duration in seconds
  "remaining": 1200             // Remaining time in seconds for current phase - ignored for "unStarted"
}
```

**Special Case for "unStarted" state:**
```json
{
  "runState": "unStarted",
  "startTime": null,      // or "<null>"
  "focusDuration": 1500,  // Still provided for context
  "restDuration": 300,    // Still provided for context
  "remaining": 0          // ignored
}
```

### Field Descriptions

- **runState**: String representing current pomodoro state
- **startTime**: Unix timestamp in **milliseconds** when the current phase started (can be `null` or `"<null>"` for "unStarted" state)
- **focusDuration**: Total duration for focus sessions in **seconds** (e.g., 1500 = 25 minutes)
- **restDuration**: Total duration for rest sessions in **seconds** (e.g., 300 = 5 minutes)
- **remaining**: Current remaining time in **seconds** for the active phase (ignored for "unStarted" state, managed by web for pause/resume)

## State Definitions

### 1. "unStarted"
- **Meaning**: Pomodoro session ended
- **Action**: Remove status item from system toolbar
- **UI**: Status item hidden

### 2. "focusing"
- **Meaning**: Focus session in progress
- **Action**: Display work countdown with blue progress ring
- **UI**: Blue circle with MM:SS countdown
- **Calculation**: Use current time vs startTime + remaining to calculate actual remaining time and progress

### 3. "focusPaused"
- **Meaning**: Focus session paused
- **Action**: Pause countdown and progress updates, save current remaining time
- **UI**: Orange circle with MM:SS (frozen at pause time + 1s correction)
- **Calculation**: Display saved paused remaining time, no progress updates
- **Resume Logic**: When resuming to "focusing", use paused remaining time as new baseline

### 4. "focusCompleted"
- **Meaning**: Focus session completed
- **Action**: Display completion state with notification
- **UI**: Blue full circle with "专注完成" text
- **Notification**: Show "Focus session completed" notification

### 5. "breaking"
- **Meaning**: Break session in progress  
- **Action**: Display break countdown with green progress ring
- **UI**: Green circle with MM:SS countdown
- **Calculation**: Use current time vs startTime + remaining to calculate actual remaining time and progress

### 6. "breakCompleted"
- **Meaning**: Break session completed
- **Action**: Display completion state with notification
- **UI**: Green full circle with "休息完成" text
- **Notification**: Show "Break session completed" notification

## State Flow Diagram

```
    ┌─────────────┐
    │  unStarted  │◄──┐
    │     (1)     │   │
    └─────────────┘   │
           │          │
           │ start    │ any state can
           ▼          │ transition to end
    ┌─────────────┐   │
    │  focusing   │───┤
    │     (2)     │   │
    └─────┬───────┘   │
          │           │
          │ pause     │
          ▼           │
    ┌─────────────┐   │
    │focusPaused  │───┤
    │     (3)     │   │
    └─────┬───────┘   │
          │           │
          │ resume    │
          ▼           │
    ┌─────────────┐   │
    │  focusing   │───┤
    │     (2)     │   │
    └─────┬───────┘   │
          │           │
          │ complete  │
          ▼           │
    ┌─────────────┐   │
    │focusCompleted│──┤
    │     (4)     │   │
    └─────┬───────┘   │
          │           │
          │ start break│
          ▼           │
    ┌─────────────┐   │
    │  breaking   │───┤
    │     (5)     │   │
    └─────┬───────┘   │
          │           │
          │ complete  │
          ▼           │
    ┌─────────────┐   │
    │breakCompleted│──┤
    │     (6)     │   │
    └─────┬───────┘   │
          │           │
          │ web decides│
          │ next state │
          └───────────┘
```

## Time Calculation Logic

### For Active States (focusing, breaking)

```
// Web provides remaining time directly, native just adds display correction
actualRemaining = remaining + 1  // +1s display correction

// Progress calculation uses appropriate duration field
totalDuration = (runState == "focusing" || runState == "focusPaused") ? focusDuration : restDuration
elapsed = totalDuration - remaining
progress = elapsed / totalDuration
```

### For Paused States (focusPaused)

```
// Web handles pause state management, native just displays
displayTime = remaining + 1  // Web-provided remaining + 1s correction
progress = (focusDuration - remaining) / focusDuration  // Progress based on focus duration
```

### Resume Logic

```
// Web component manages all pause/resume logic:
// 1. Web calculates remaining time during pause
// 2. Web provides updated remaining when resuming
// 3. Native simply displays the web-provided values
// 4. No native pause time tracking needed
```

### For Completed States

```
displayTime = "00:00" or completion text
progress = 1.0 (full circle)
```

## UI Display Strategy

| State | Status Bar Display | Color | Text |
|-------|-------------------|-------|------|
| "unStarted" | Hidden | - | - |
| "focusing" | [🔵●○○○○] MM:SS | Blue | Countdown |
| "focusPaused" | [🟠●○○○○] MM:SS | Orange | Frozen time |
| "focusCompleted" | [🔵●●●●●] | Blue | 专注完成 |
| "breaking" | [🟢●○○○○] MM:SS | Green | Countdown |
| "breakCompleted" | [🟢●●●●●] | Green | 休息完成 |

## Implementation Notes

1. **Web-Driven State Management**: All state transitions are controlled by the web component
2. **No Native Timers**: Native code doesn't trigger state changes based on time expiration
3. **Real-time Updates**: For active states, update display every second using web-provided remaining time
4. **Web-Managed Pause/Resume**: Web component handles all pause timing logic, native just displays current state
5. **Duration-Based Progress**: Use focusDuration for focus states, restDuration for break states
6. **Flexible Next State**: After completion, web component determines next state (could be any valid state)

## Migration from Old System

The old system used:
- `state`: String values ("started", "paused", "stopped", "resumed")
- `type`: String values ("work", "break") 
- `duration`: Total duration
- Manual state management with pausedDuration tracking

The new system:
- Uses string-based runState values
- Provides remaining time directly from web
- Includes focusDuration and restDuration for progress calculation
- Web handles all pause/resume timing logic
- Eliminates need for native pause duration calculation
- Simplifies native state management to display-only logic