# Pomodoro State Manager Test Scenarios

## Test Data Structure

All test scenarios use the new notification format:

```json
{
  "runState": number,    // 1-6
  "startTime": number,   // Unix timestamp
  "pauseTime": number,   // Currently unused
  "remaining": number    // Seconds remaining
}
```

## Test Scenario Categories

### 1. Basic State Transitions

#### Test 1.1: Start Focus Session
**Input:**
```json
{
  "runState": 2,
  "startTime": 1702234800,
  "pauseTime": 0,
  "remaining": 1500
}
```
**Expected Behavior:**
- Status item appears in menu bar
- Blue circle with progress starts at 0%
- Countdown shows 25:00
- Timer updates every second

#### Test 1.2: Pause Focus Session
**Prerequisites:** Focus session active
**Input:**
```json
{
  "runState": 3,
  "startTime": 1702234800,
  "pauseTime": 0,
  "remaining": 1200
}
```
**Expected Behavior:**
- Circle changes to orange
- Time display freezes at current value (20:00)
- Timer stops updating

#### Test 1.3: Resume Focus Session
**Prerequisites:** Focus session paused
**Input:**
```json
{
  "runState": 2,
  "startTime": 1702235000,
  "pauseTime": 0,
  "remaining": 1200
}
```
**Expected Behavior:**
- Circle changes back to blue
- Time resumes counting down from paused value
- Timer updates restart

#### Test 1.4: Complete Focus Session
**Prerequisites:** Focus session active
**Input:**
```json
{
  "runState": 4,
  "startTime": 1702234800,
  "pauseTime": 0,
  "remaining": 0
}
```
**Expected Behavior:**
- Circle becomes full blue
- Text changes to "专注完成"
- System notification appears: "You've finished 1 Pomodoro. Take a break."
- Timer stops

#### Test 1.5: Start Break Session
**Prerequisites:** Focus completed
**Input:**
```json
{
  "runState": 5,
  "startTime": 1702236300,
  "pauseTime": 0,
  "remaining": 300
}
```
**Expected Behavior:**
- Circle changes to green
- Progress starts at 0%
- Countdown shows 05:00
- Timer updates every second

#### Test 1.6: Complete Break Session
**Prerequisites:** Break session active
**Input:**
```json
{
  "runState": 6,
  "startTime": 1702236300,
  "pauseTime": 0,
  "remaining": 0
}
```
**Expected Behavior:**
- Circle becomes full green
- Text changes to "休息完成"
- System notification appears: "Break time is over. Ready to work!"
- Timer stops

#### Test 1.7: End Session
**Prerequisites:** Any active state
**Input:**
```json
{
  "runState": 1,
  "startTime": 0,
  "pauseTime": 0,
  "remaining": 0
}
```
**Expected Behavior:**
- Status item disappears from menu bar
- All timers stop
- State resets to idle

### 2. Time Calculation Tests

#### Test 2.1: Real-time Countdown Accuracy
**Setup:**
- Start focus session with 60 seconds remaining
- Wait 10 seconds
- Check display

**Expected:**
- Display should show 00:50 after 10 seconds
- Progress should be ~16.7% (10/60)

#### Test 2.2: Pause Time Preservation
**Setup:**
- Focus session with 30 seconds remaining
- Pause session
- Wait 10 seconds
- Check display

**Expected:**
- Display should still show 00:30 (frozen)
- Progress should remain unchanged

#### Test 2.3: Resume Time Accuracy
**Setup:**
- Resume from above paused state
- New startTime = current time
- Remaining = 30 seconds

**Expected:**
- Countdown should resume from 00:30
- Progress calculation should be accurate

### 3. Edge Cases

#### Test 3.1: Invalid runState Values
**Input:**
```json
{
  "runState": 0,  // Invalid
  "startTime": 1702234800,
  "pauseTime": 0,
  "remaining": 1500
}
```
**Expected:** Notification ignored, no state change

#### Test 3.2: Negative Remaining Time
**Input:**
```json
{
  "runState": 2,
  "startTime": 1702234800,
  "pauseTime": 0,
  "remaining": -100  // Invalid
}
```
**Expected:** Notification ignored, no state change

#### Test 3.3: Missing Required Fields
**Input:**
```json
{
  "runState": 2,
  "startTime": 1702234800
  // Missing remaining field
}
```
**Expected:** Notification ignored, no state change

#### Test 3.4: Very Large Remaining Time
**Input:**
```json
{
  "runState": 2,
  "startTime": 1702234800,
  "pauseTime": 0,
  "remaining": 86400  // 24 hours
}
```
**Expected:** Notification ignored (exceeds 2-hour limit)

### 4. UI Display Tests

#### Test 4.1: Progress Ring Accuracy
**Setup:** Focus session with 25 minutes total

| Elapsed | Expected Progress | Expected Display |
|---------|------------------|------------------|
| 0 min   | 0%               | 25:00           |
| 5 min   | 20%              | 20:00           |
| 12.5 min| 50%              | 12:30           |
| 20 min  | 80%              | 05:00           |
| 25 min  | 100%             | 专注完成         |

#### Test 4.2: Color Transitions
**Sequence:**
1. Focus (Blue) → Pause (Orange) → Resume (Blue)
2. Focus Complete (Blue Full) → Break (Green) → Break Complete (Green Full)

#### Test 4.3: Text Display Formatting
- Time format: MM:SS with leading zeros
- Completion text: Proper Chinese characters
- Font: Monospaced for consistent width

### 5. Integration Tests

#### Test 5.1: Context Menu Functionality
**Prerequisites:** Any active pomodoro state
- Right-click status item
- Verify "Open main window" works
- Verify "Exit Pomodoro" sends proper exit command

#### Test 5.2: Window Integration
**Prerequisites:** Main window closed
- Click "Open main window" from status menu
- Verify window opens and shows pomodoro view

#### Test 5.3: Notification Permissions
**Setup:** First completion event
- Verify system asks for notification permission
- Test both granted and denied scenarios

### 6. Performance Tests

#### Test 6.1: Timer Performance
**Setup:** Long-running session (1 hour)
- Monitor CPU usage during countdown
- Verify smooth 1-second updates
- Check memory usage stability

#### Test 6.2: Multiple State Changes
**Setup:** Rapid state transitions
- Send 10 state changes within 1 second
- Verify final state is correct
- Check for race conditions

### 7. Stress Tests

#### Test 7.1: Invalid Data Flood
**Setup:** Send 100 invalid notifications rapidly
- System should remain stable
- Valid notifications should still work
- No memory leaks

#### Test 7.2: State Change During Update
**Setup:** Send state change during display update
- No crashes or inconsistent state
- Display should reflect final state

## Test Environment Setup

### Mock Data Generator
```javascript
// For testing in web component
function sendMockNotification(runState, remaining) {
  const notification = {
    runState: runState,
    startTime: Math.floor(Date.now() / 1000),
    pauseTime: 0,
    remaining: remaining
  };
  // Send to native component
}
```

### Test Automation
- Create test harness for automated state transitions
- Implement time simulation for faster testing
- Add logging for debugging state changes

## Success Criteria

1. All basic state transitions work correctly
2. Time calculations are accurate within 1 second
3. UI updates smoothly without flicker
4. Edge cases are handled gracefully
5. No memory leaks during long sessions
6. Integration with existing UI components works
7. Performance remains acceptable under load