# YX-182 番茄钟状态栏功能 - 最终实现方案

## 方案选择：非模板图像 + 自定义绘制

基于技术验证，我们采用 **非模板图像** 的方案来实现状态栏的颜色控制：

### 核心优势
- ✅ **完美颜色控制**: 通过 `isTemplate = false` 保持自定义颜色
- ✅ **实现简单**: 直接绘制 NSImage，无需复杂的视图管理
- ✅ **性能优良**: 图像绘制后缓存，更新频率可控
- ✅ **兼容性好**: 适用于所有 macOS 版本

## 技术架构

### 1. 文件结构（简化版）
```
Source/Application/Pomodoro-Status/
├── ENPomodoroStatusManager.h          // 主管理类
├── ENPomodoroStatusManager.m
├── ENPomodoroIconRenderer.h           // 图标绘制工具
└── ENPomodoroIconRenderer.m
```

### 2. 核心类设计

#### ENPomodoroStatusManager
```objc
@interface ENPomodoroStatusManager : NSObject

@property (weak, nonnull) ENAccountController *accountController;
@property (readonly) BOOL isActive;

// 主要接口
- (nonnull instancetype)initWithAccountController:(nonnull ENAccountController *)accountController;
- (void)handleStateChangeNotification:(nonnull NSDictionary *)info;
- (void)showStatusItem;
- (void)hideStatusItem;

@end
```

#### ENPomodoroIconRenderer
```objc
@interface ENPomodoroIconRenderer : NSObject

+ (nonnull NSImage *)imageWithProgress:(CGFloat)progress
                             ringColor:(nonnull NSColor *)ringColor
                               bgColor:(nonnull NSColor *)bgColor
                                  size:(CGFloat)size;

+ (nonnull NSImage *)workImageWithProgress:(CGFloat)progress size:(CGFloat)size;
+ (nonnull NSImage *)breakImageWithProgress:(CGFloat)progress size:(CGFloat)size;

@end
```

## 实现细节

### 1. 图标绘制逻辑
```objc
// ENPomodoroIconRenderer.m
+ (NSImage *)imageWithProgress:(CGFloat)progress
                     ringColor:(NSColor *)ringColor
                       bgColor:(NSColor *)bgColor
                          size:(CGFloat)size {
    
    NSSize imageSize = NSMakeSize(size, size);
    NSImage *image = [[NSImage alloc] initWithSize:imageSize];
    
    [image lockFocus];
    
    CGFloat lineWidth = 2.0;
    NSRect rect = NSMakeRect(lineWidth/2, lineWidth/2, 
                           size-lineWidth, size-lineWidth);
    NSPoint center = NSMakePoint(NSMidX(rect), NSMidY(rect));
    CGFloat radius = rect.size.width / 2;
    
    // 背景圆环
    NSBezierPath *bgPath = [NSBezierPath bezierPath];
    [bgPath appendBezierPathWithArcWithCenter:center
                                       radius:radius
                                   startAngle:0
                                     endAngle:360
                                    clockwise:NO];
    bgPath.lineWidth = lineWidth;
    [bgColor setStroke];
    [bgPath stroke];
    
    // 进度圆环（从顶部顺时针）
    if (progress > 0) {
        CGFloat angle = progress * 360.0;
        CGFloat startAngle = 90;
        CGFloat endAngle = 90 - angle;
        
        NSBezierPath *progressPath = [NSBezierPath bezierPath];
        [progressPath appendBezierPathWithArcWithCenter:center
                                                 radius:radius
                                             startAngle:startAngle
                                               endAngle:endAngle
                                              clockwise:YES];
        progressPath.lineCapStyle = NSLineCapStyleRound;
        progressPath.lineWidth = lineWidth;
        [ringColor setStroke];
        [progressPath stroke];
    }
    
    [image unlockFocus];
    
    // 关键：不使用模板图像
    image.template = NO;
    
    return image;
}

// 便利方法
+ (NSImage *)workImageWithProgress:(CGFloat)progress size:(CGFloat)size {
    return [self imageWithProgress:progress
                         ringColor:[NSColor labelColor]  // 工作：黑色
                           bgColor:[[NSColor systemGray] colorWithAlphaComponent:0.3]
                              size:size];
}

+ (NSImage *)breakImageWithProgress:(CGFloat)progress size:(CGFloat)size {
    return [self imageWithProgress:progress  
                         ringColor:[NSColor systemGreenColor]  // 休息：绿色
                           bgColor:[[NSColor systemGray] colorWithAlphaComponent:0.3]
                              size:size];
}
```

### 2. 状态管理器核心逻辑
```objc
// ENPomodoroStatusManager.m
@interface ENPomodoroStatusManager ()
@property (strong) NSStatusItem *statusItem;
@property (strong) NSTimer *updateTimer;
@property (strong) NSMenu *contextMenu;

// 状态数据
@property (assign) NSTimeInterval startTime;
@property (assign) NSTimeInterval duration;  
@property (assign) NSTimeInterval pausedDuration;
@property (assign) BOOL isBreakTime;
@property (assign) BOOL isPaused;
@end

@implementation ENPomodoroStatusManager

- (void)handleStateChangeNotification:(NSDictionary *)info {
    if (![self validateNotificationInfo:info]) return;
    
    NSString *state = info[@"state"];
    self.startTime = [info[@"startTime"] doubleValue];
    self.duration = [info[@"duration"] doubleValue];
    self.isBreakTime = [info[@"type"] isEqualToString:@"break"];
    
    if ([state isEqualToString:@"started"]) {
        [self startPomodoro];
    } else if ([state isEqualToString:@"paused"]) {
        [self pausePomodoro];
    } else if ([state isEqualToString:@"resumed"]) {
        [self resumePomodoro];
    } else if ([state isEqualToString:@"stopped"]) {
        [self stopPomodoro];
    }
}

- (void)startPomodoro {
    [self showStatusItem];
    [self startUpdateTimer];
    self.pausedDuration = 0;
    self.isPaused = NO;
    [self updateDisplay];
}

- (void)updateDisplay {
    NSTimeInterval now = [[NSDate date] timeIntervalSince1970];
    NSTimeInterval elapsed = now - self.startTime - self.pausedDuration;
    NSTimeInterval remaining = MAX(0, self.duration - elapsed);
    CGFloat progress = (self.duration - remaining) / self.duration;
    
    // 格式化时间显示
    NSInteger minutes = (NSInteger)(remaining / 60);
    NSInteger seconds = (NSInteger)remaining % 60;
    NSString *timeText = [NSString stringWithFormat:@"%02ld:%02ld", minutes, seconds];
    
    // 生成图标
    NSImage *icon;
    if (self.isBreakTime) {
        icon = [ENPomodoroIconRenderer breakImageWithProgress:progress size:18];
    } else {
        icon = [ENPomodoroIconRenderer workImageWithProgress:progress size:18];
    }
    
    // 更新状态栏
    NSStatusBarButton *button = self.statusItem.button;
    button.image = icon;
    button.imagePosition = NSImageLeft;
    button.title = timeText;
    
    // 检查是否完成
    if (remaining <= 0) {
        [self pomodoroCompleted];
    }
}

- (void)startUpdateTimer {
    [self.updateTimer invalidate];
    self.updateTimer = [NSTimer scheduledTimerWithTimeInterval:1.0
                                                        target:self
                                                      selector:@selector(updateDisplay)
                                                      userInfo:nil
                                                       repeats:YES];
    [[NSRunLoop mainRunLoop] addTimer:self.updateTimer forMode:NSRunLoopCommonModes];
}

@end
```

### 3. 右键菜单实现
```objc
- (void)setupContextMenu {
    self.contextMenu = [[NSMenu alloc] init];
    
    NSMenuItem *openItem = [[NSMenuItem alloc] 
        initWithTitle:NSLocalizedString(@"Open main window", @"Pomodoro menu")
               action:@selector(openMainWindow:)
        keyEquivalent:@""];
    openItem.target = self;
    
    NSMenuItem *exitItem = [[NSMenuItem alloc]
        initWithTitle:NSLocalizedString(@"Exit Pomodoro", @"Pomodoro menu") 
               action:@selector(exitPomodoro:)
        keyEquivalent:@""];
    exitItem.target = self;
    
    [self.contextMenu addItem:openItem];
    [self.contextMenu addItem:[NSMenuItem separatorItem]];
    [self.contextMenu addItem:exitItem];
}

- (void)setupStatusItemActions {
    NSStatusBarButton *button = self.statusItem.button;
    button.target = self;
    button.action = @selector(statusItemClicked:);
    
    // 同时响应左键和右键
    [button sendActionOn:NSEventMaskLeftMouseUp | NSEventMaskRightMouseUp];
}

- (void)statusItemClicked:(NSStatusBarButton *)sender {
    NSEvent *event = [NSApp currentEvent];
    
    if (event.type == NSEventTypeRightMouseUp) {
        // 右键：显示菜单
        self.statusItem.menu = self.contextMenu;
        [sender performClick:nil];
        self.statusItem.menu = nil;  // 重要：用完清除避免影响左键
    } else {
        // 左键：可以添加其他功能，比如暂停/恢复
        [self togglePause];
    }
}
```

## 颜色方案

### 1. 颜色定义
- **工作状态**: `NSColor.labelColor` (适配深色/浅色模式的黑色)
- **休息状态**: `NSColor.systemGreenColor` (系统绿色)
- **暂停状态**: `NSColor.systemOrangeColor` (橙色)
- **背景圆环**: `NSColor.systemGray` alpha 0.3 (半透明灰色)

### 2. 深色模式适配
由于使用了系统颜色 (`NSColor.labelColor`, `NSColor.systemGreenColor`)，会自动适配深色模式。

## 集成步骤

### 1. 添加到 ENAccountController
```objc
// ENAccountController.h
@property (strong, nonnull, readonly) ENPomodoroStatusManager *pomodoroStatusManager;

// ENAccountController.m 
- (instancetype)initWithAccountDecorator:(ENMacEvernoteAccountDecorator *)accountDecorator {
    // ... 现有代码
    _pomodoroStatusManager = [[ENPomodoroStatusManager alloc] initWithAccountController:self];
    // ...
}
```

### 2. 连接 PomodoroViewController
```swift
// PomodoroViewController.swift
func handleNotificationStateChange(_ info: [String: Any]) {
    AppLogInfo("[Pomodoro] State change: \(info)")
    
    guard let accountController = ENAccountControllersManager.shared().mainAccountController else {
        return
    }
    
    accountController.pomodoroStatusManager.handleStateChangeNotification(info)
}
```

## 最终效果

- 📱 状态栏显示：`🟢 25:00` (圆环 + 时间)
- 🎨 颜色区分：黑色圆环=工作，绿色圆环=休息  
- 🖱 右键菜单：打开主窗口 / 退出番茄钟
- ⏱ 实时更新：每秒更新进度和时间
- 🌓 主题适配：自动适配深色/浅色模式

这个方案既实用又美观，完全符合需求并且实现简单可靠。