# 番茄钟状态栏颜色实现方案

## 背景问题

macOS 状态栏对自定义颜色的支持有以下限制：
1. 系统会自动适配状态栏图标的颜色（深色/浅色模式）
2. 使用自定义视图需要处理32+种样式变化
3. Apple 推荐使用模板图像让系统处理颜色适配

## 推荐方案：动态模板图像

### 实现思路
1. 动态生成带有圆环和时间的 NSImage
2. 根据番茄钟状态选择合适的基础颜色
3. 设置为模板图像，让系统处理最终显示效果
4. 定期更新图像内容

### 核心代码结构

```objc
@interface ENPomodoroStatusManager : NSObject
- (void)updateStatusItemWithTime:(NSTimeInterval)remainingTime
                       totalTime:(NSTimeInterval)totalTime
                     isBreakTime:(BOOL)isBreakTime;
@end

@implementation ENPomodoroStatusManager

- (void)updateStatusItemWithTime:(NSTimeInterval)remainingTime 
                       totalTime:(NSTimeInterval)totalTime
                     isBreakTime:(BOOL)isBreakTime {
    
    // 格式化时间
    NSString *timeString = [self formatTime:remainingTime];
    
    // 计算进度
    CGFloat progress = (totalTime - remainingTime) / totalTime;
    
    // 生成图像
    NSImage *statusImage = [self createStatusImageWithTime:timeString
                                                  progress:progress
                                               isBreakTime:isBreakTime];
    
    // 更新状态栏
    self.statusItem.button.image = statusImage;
}

- (NSImage *)createStatusImageWithTime:(NSString *)timeString
                              progress:(CGFloat)progress
                           isBreakTime:(BOOL)isBreakTime {
    
    NSSize imageSize = NSMakeSize(50, 22); // 状态栏合适尺寸
    NSImage *image = [[NSImage alloc] initWithSize:imageSize];
    
    [image lockFocus];
    
    // 设置绘制属性
    [[NSGraphicsContext currentContext] setShouldAntialias:YES];
    
    // 绘制圆环
    [self drawProgressRingWithProgress:progress 
                            isBreakTime:isBreakTime 
                               inRect:NSMakeRect(2, 3, 16, 16)];
    
    // 绘制时间文本
    [self drawTimeText:timeString 
           isBreakTime:isBreakTime 
                inRect:NSMakeRect(20, 0, 28, 22)];
    
    [image unlockFocus];
    
    // 关键：设置为模板图像
    [image setTemplate:YES];
    
    return image;
}

- (void)drawProgressRingWithProgress:(CGFloat)progress
                         isBreakTime:(BOOL)isBreakTime
                              inRect:(NSRect)rect {
    
    NSPoint center = NSMakePoint(NSMidX(rect), NSMidY(rect));
    CGFloat radius = MIN(rect.size.width, rect.size.height) / 2 - 1;
    
    // 背景圆环
    NSBezierPath *backgroundPath = [NSBezierPath bezierPath];
    [backgroundPath appendBezierPathWithArcWithCenter:center
                                               radius:radius
                                           startAngle:0
                                             endAngle:360];
    [backgroundPath setLineWidth:1.5];
    [[NSColor colorWithWhite:0.0 alpha:0.3] setStroke]; // 淡色背景
    [backgroundPath stroke];
    
    // 进度圆环
    if (progress > 0) {
        NSBezierPath *progressPath = [NSBezierPath bezierPath];
        CGFloat startAngle = 90; // 从顶部开始
        CGFloat endAngle = startAngle - (progress * 360); // 顺时针
        
        [progressPath appendBezierPathWithArcWithCenter:center
                                                 radius:radius
                                             startAngle:startAngle
                                               endAngle:endAngle
                                              clockwise:YES];
        [progressPath setLineWidth:1.5];
        
        // 使用全黑色，让模板图像系统处理最终颜色
        [[NSColor blackColor] setStroke];
        [progressPath stroke];
    }
}

- (void)drawTimeText:(NSString *)timeString
         isBreakTime:(BOOL)isBreakTime
              inRect:(NSRect)rect {
    
    // 文本属性
    NSMutableDictionary *attributes = [NSMutableDictionary dictionary];
    attributes[NSFontAttributeName] = [NSFont monospacedDigitSystemFontOfSize:9.0 weight:NSFontWeightMedium];
    
    // 使用黑色，让模板图像系统处理颜色
    attributes[NSForegroundColorAttributeName] = [NSColor blackColor];
    
    // 计算文本位置（居中）
    NSSize textSize = [timeString sizeWithAttributes:attributes];
    NSRect textRect = NSMakeRect(
        rect.origin.x + (rect.size.width - textSize.width) / 2,
        rect.origin.y + (rect.size.height - textSize.height) / 2,
        textSize.width,
        textSize.height
    );
    
    [timeString drawInRect:textRect withAttributes:attributes];
}

- (NSString *)formatTime:(NSTimeInterval)timeInterval {
    int minutes = (int)(timeInterval / 60);
    int seconds = (int)timeInterval % 60;
    return [NSString stringWithFormat:@"%d:%02d", minutes, seconds];
}

@end
```

## 颜色处理策略

### 1. 模板图像的优势
- **系统自适应**: 自动适配深色/浅色模式
- **一致性**: 与其他状态栏图标保持视觉一致
- **简化实现**: 不需要手动处理外观变化

### 2. 颜色区分方案

由于模板图像会被系统重新着色，我们需要用其他方式区分工作/休息状态：

#### 方案A: 图形区分
```objc
// 工作状态：实心圆环
- (void)drawWorkRingWithProgress:(CGFloat)progress inRect:(NSRect)rect {
    // 绘制实心进度条
}

// 休息状态：虚线圆环  
- (void)drawBreakRingWithProgress:(CGFloat)progress inRect:(NSRect)rect {
    NSBezierPath *path = [NSBezierPath bezierPath];
    // 设置虚线模式
    CGFloat dashPattern[] = {2.0, 2.0};
    [path setLineDash:dashPattern count:2 phase:0];
    // 绘制虚线进度条
}
```

#### 方案B: 符号区分  
```objc
// 工作状态：显示 "W" 或工作图标
// 休息状态：显示 "B" 或休息图标
- (void)drawStateIndicator:(BOOL)isBreakTime inRect:(NSRect)rect {
    NSString *indicator = isBreakTime ? @"●" : @"■"; // 不同符号
    // 绘制状态指示符
}
```

#### 方案C: 动画区分
```objc
// 工作状态：正常圆环
// 休息状态：轻微闪烁或渐变效果
- (void)startBreakAnimation {
    if (self.isBreakTime) {
        // 定期切换透明度实现闪烁
        [NSTimer scheduledTimerWithTimeInterval:1.0 repeats:YES block:^(NSTimer *timer) {
            // 更新图像透明度
        }];
    }
}
```

## 最终推荐方案

结合实用性和视觉效果，推荐使用：

1. **基础**: 动态生成模板图像
2. **区分**: 虚线圆环区分休息状态 
3. **文本**: 统一使用 MM:SS 格式
4. **更新**: 每秒更新一次图像

### 优势
- ✅ 完美适配系统主题
- ✅ 实现简单，维护容易  
- ✅ 性能良好，资源占用小
- ✅ 视觉效果清晰易识别

### 实施要点
1. 图像尺寸控制在 50x22 像素内
2. 使用等宽字体确保时间显示稳定
3. 圆环线条不要太细，确保在小尺寸下可见
4. 定期清理图像缓存避免内存泄漏

这个方案既满足了颜色需求，又保证了与 macOS 系统的兼容性。