# YX-182 番茄钟状态栏功能设计方案

## 1. 需求概述

在系统菜单栏显示番茄钟状态，包含圆环进度条和剩余时间显示，支持右键菜单操作。

### 核心功能
- 状态栏显示圆环 + 剩余时间（MM:SS 格式）
- 支持自定义颜色：番茄钟时间为黑色，休息时间为绿色
- 右键菜单：Open main window、Exit Pomodoro
- 番茄钟完成时显示通知弹窗
- 多状态支持：开始/暂停/恢复/停止

## 2. 技术架构

### 2.1 核心组件设计

```
ENAccountController
    └── ENPomodoroStatusManager (主管理类)
        ├── NSStatusItem (状态栏项)
        ├── ENPomodoroStatusView (自定义视图)
        ├── NSMenu (右键菜单)
        ├── NSPopover (完成通知)
        └── NSTimer (倒计时器)
```

### 2.2 通知数据结构

基于 `handleNotificationStateChange` 接收的数据：

```swift
// 通知信息结构
struct PomodoroNotificationInfo {
    let startTime: TimeInterval      // Unix 时间戳
    let type: PomodoroType          // .work / .break
    let duration: TimeInterval      // 持续时长（秒）
    let state: PomodoroState        // .started / .paused / .stopped / .resumed
}

enum PomodoroType: String {
    case work = "work"
    case break = "break"
}

enum PomodoroState: String {
    case started = "started"
    case paused = "paused" 
    case stopped = "stopped"
    case resumed = "resumed"
}
```

## 3. 类设计详解

### 3.1 ENPomodoroStatusManager

**职责：**
- 管理状态栏的显示/隐藏
- 处理番茄钟状态变更通知
- 管理倒计时逻辑
- 控制右键菜单和弹窗

**主要接口：**
```objc
@interface ENPomodoroStatusManager : NSObject

@property (weak, nonnull) ENAccountController *accountController;
@property (readonly) BOOL isActive;

// 状态栏控制
- (void)showStatusItem;
- (void)hideStatusItem;

// 番茄钟状态处理
- (void)handleStateChangeNotification:(NSDictionary *)info;

// 内部方法
- (void)updateTimerWithStartTime:(NSTimeInterval)startTime 
                        duration:(NSTimeInterval)duration 
                            type:(NSString *)type 
                           state:(NSString *)state;

@end
```

### 3.2 ENPomodoroStatusView

**职责：**
- 绘制圆环进度条
- 显示剩余时间文本
- 响应鼠标事件

**视觉规格：**
- 整体尺寸：24x22 points（符合 macOS 状态栏标准）
- 圆环直径：16 points
- 线条宽度：2 points
- 字体：System font 9pt
- 颜色：
  - 工作时间：`NSColor.labelColor`（黑色/白色适应系统主题）
  - 休息时间：`NSColor.systemGreen`

**绘制逻辑：**
```objc
- (void)drawRect:(NSRect)dirtyRect {
    // 1. 计算进度角度
    CGFloat progress = (totalTime - remainingTime) / totalTime;
    CGFloat angle = progress * 2 * M_PI;
    
    // 2. 绘制背景圆环
    // 3. 绘制进度圆环
    // 4. 绘制时间文本
}
```

### 3.3 数据流设计

```
PomodoroViewController (Web)
    ↓ handleNotificationStateChange
ENAccountController.pomodoroStatusManager
    ↓ handleStateChangeNotification
ENPomodoroStatusManager
    ↓ updateTimer
NSTimer (每秒触发)
    ↓ updateDisplay
ENPomodoroStatusView
    ↓ drawRect
视觉更新
```

## 4. 状态机设计

### 4.1 状态定义

```objc
typedef NS_ENUM(NSInteger, ENPomodoroManagerState) {
    ENPomodoroManagerStateIdle,      // 空闲状态
    ENPomodoroManagerStateActive,    // 活跃运行
    ENPomodoroManagerStatePaused,    // 暂停状态
};
```

### 4.2 状态转换

```
Idle → Active: 收到 started 通知
Active → Paused: 收到 paused 通知  
Paused → Active: 收到 resumed 通知
Any → Idle: 收到 stopped 通知或倒计时结束
```

## 5. 用户交互设计

### 5.1 右键菜单

```objc
- (NSMenu *)createContextMenu {
    NSMenu *menu = [[NSMenu alloc] init];
    
    // "打开主窗口"
    NSMenuItem *openItem = [[NSMenuItem alloc] 
        initWithTitle:NSLocalizedString(@"Open main window", @"Pomodoro status menu")
        action:@selector(openMainWindow:)
        keyEquivalent:@""];
    openItem.target = self;
    
    // "退出番茄钟"  
    NSMenuItem *exitItem = [[NSMenuItem alloc]
        initWithTitle:NSLocalizedString(@"Exit Pomodoro", @"Pomodoro status menu")
        action:@selector(exitPomodoro:)
        keyEquivalent:@""];
    exitItem.target = self;
    
    [menu addItem:openItem];
    [menu addItem:[NSMenuItem separatorItem]];
    [menu addItem:exitItem];
    
    return menu;
}
```

### 5.2 完成通知弹窗

使用 `NSPopover` 在状态栏下方显示：

```objc
- (void)showCompletionNotification {
    NSViewController *popoverVC = [self createCompletionPopoverViewController];
    
    NSPopover *popover = [[NSPopover alloc] init];
    popover.contentViewController = popoverVC;
    popover.behavior = NSPopoverBehaviorTransient;
    
    [popover showRelativeToRect:self.statusView.bounds 
                         ofView:self.statusView 
                  preferredEdge:NSMinYEdge];
    
    // 3秒后自动消失
    dispatch_after(dispatch_time(DISPATCH_TIME_NOW, 3 * NSEC_PER_SEC), 
                   dispatch_get_main_queue(), ^{
        [popover close];
    });
}
```

## 6. 集成方案

### 6.1 ENAccountController 集成

**头文件修改：**
```objc
// ENAccountController.h
@class ENPomodoroStatusManager;

@interface ENAccountController : NSResponder
// 在现有属性后添加
@property (strong, nonnull, readonly) ENPomodoroStatusManager *pomodoroStatusManager;
@end
```

**实现文件修改：**
```objc
// ENAccountController.m
- (instancetype)initWithAccountDecorator:(ENMacEvernoteAccountDecorator *)accountDecorator {
    // 在现有初始化代码后添加
    _pomodoroStatusManager = [[ENPomodoroStatusManager alloc] initWithAccountController:self];
}

- (void)tearDown {
    // 在现有清理代码中添加
    [_pomodoroStatusManager hideStatusItem];
}
```

### 6.2 PomodoroViewController 集成

```swift
// PomodoroViewController.swift
func handleNotificationStateChange(_ info: [String: Any]) {
    AppLogInfo("[Pomodoro] State change: \(info)")
    
    // 获取当前账户的状态管理器
    if let accountController = ENAccountControllersManager.shared().mainAccountController {
        accountController.pomodoroStatusManager.handleStateChangeNotification(info)
    }
}
```

### 6.3 主窗口调用集成

```objc
// ENPomodoroStatusManager.m
- (void)openMainWindow:(id)sender {
    // 获取主窗口控制器
    ENNoteCollectionWindowController *mainWindow = [ENAccountControllersManager.sharedManager.mainAccountController.mainNoteCollectionWindowController];
    
    if (mainWindow) {
        [mainWindow.window makeKeyAndOrderFront:nil];
        [mainWindow showPomodoro];
    }
}
```

## 7. 国际化支持

创建本地化字符串：

```objc
// 在适当的 .strings 文件中添加
"Open main window" = "打开主窗口";
"Exit Pomodoro" = "退出番茄钟";
"You've finished 1 Pomodoro. Take a break." = "您已完成1个番茄钟，请休息一下。";
"Work time" = "工作时间";
"Break time" = "休息时间";
```

## 8. 错误处理和边界情况

### 8.1 网络异常处理
- Web 页面断开连接时，保持当前倒计时状态
- 重新连接后同步状态

### 8.2 系统休眠处理
- 系统休眠时暂停倒计时
- 系统唤醒后询问是否继续

### 8.3 多实例处理
- 确保同时只有一个状态栏项
- 新账户登录时正确切换管理器

## 9. 性能优化

### 9.1 内存管理
- 使用 weak 引用避免循环引用
- 及时释放 NSTimer 和通知监听

### 9.2 CPU 优化
- 仅在可见时更新绘制
- 减少不必要的重绘操作

### 9.3 电池优化
- 在暂停状态下停止定时器
- 使用系统标准的更新频率

这个设计方案提供了完整的技术实现路径，确保功能完整性和用户体验的一致性。