# Whiteboard远程HTML服务器架构原理图与实现方案

## 📋 概述

本文档详细描述了Whiteboard WebView支持远程HTML服务器的架构设计、数据传输流程和实现方案。该方案通过本地HTTP代理服务器解决跨域问题，同时支持evernotecid://自定义协议的访问。

## 🎯 远程HTML服务器架构原理图

```
┌─────────────────────────────────────────────────────────────────────────────────────┐
│                              Evernote Mac App                                      │
│                                                                                     │
│  ┌─────────────────────────┐    ┌─────────────────────────────────────────────────┐ │
│  │    ENWhiteboardEditor   │    │         ENWhiteboardHTTPServer                  │ │
│  │      WKWebView          │    │                                                 │ │
│  │                         │    │  ┌─────────────┐  ┌─────────────────────────┐   │ │
│  │  ┌───────────────────┐  │    │  │   Static    │  │     API Proxy           │   │ │
│  │  │                   │  │◄───┤  │   Proxy     │  │  /api/evernotecid/*     │   │ │
│  │  │   HTML + JS       │  │    │  │   Handler   │  │                         │   │ │
│  │  │                   │  │    │  └─────────────┘  └─────────────────────────┘   │ │
│  │  └───────────────────┘  │    │          │                      │               │ │
│  │                         │    │          ▼                      ▼               │ │
│  └─────────────────────────┘    └──────────────────────────────────────────────────┘ │
│              │                            │                      │                   │
└──────────────┼────────────────────────────┼──────────────────────┼───────────────────┘
               │                            │                      │
               │ ① HTTP Request             │ ② Remote Fetch       │ ③ evernotecid Request
               │                            │                      │
               ▼                            ▼                      ▼
    ┌─────────────────┐          ┌─────────────────────┐  ┌─────────────────────┐
    │                 │          │                     │  │                     │
    │ localhost:54321 │          │  10.228.32.31:8888  │  │   ENContentProtocol │
    │                 │          │                     │  │                     │
    │ [Local Proxy]   │          │  [Remote Server]    │  │  [Local Protocol]   │
    │                 │          │                     │  │                     │
    └─────────────────┘          └─────────────────────┘  └─────────────────────┘
```

## 📋 数据传输流程详解

### 1️⃣ HTML页面加载流程
```
WebView Request: http://localhost:54321/index.html
        │
        ▼
┌─────────────────────────────────────┐
│     Local HTTP Server              │
│                                     │
│ if (remoteServerURL exists) {       │
│   ┌─────────────────────────────┐   │
│   │ Remote Proxy Handler        │   │
│   │                             │   │
│   │ 1. Fetch from remote:       │   │
│   │    http://10.228.32.31:8888/│   │◄────── ② Remote Fetch
│   │    index.html               │   │
│   │                             │   │
│   │ 2. Inject URL rewrite script│   │
│   │    into <head> tag          │   │
│   │                             │   │
│   │ 3. Return modified HTML     │   │
│   └─────────────────────────────┘   │
│ }                                   │
└─────────────────────────────────────┘
        │
        ▼ ① Modified HTML Response
     WebView
```

### 2️⃣ 静态资源加载流程
```
JavaScript Request: fetch('/superTable.js')
        │
        ▼
┌─────────────────────────────────────┐
│     Local HTTP Server              │
│                                     │
│ Static Resource Handler             │
│                                     │
│ if (remoteServerURL exists) {       │
│   URL: http://10.228.32.31:8888/   │◄────── Remote Fetch
│        superTable.js               │
│ } else {                            │
│   Local File: common-database-mac/ │
│               superTable.js        │
│ }                                   │
└─────────────────────────────────────┘
        │
        ▼ Static Resource Response
     WebView
```

### 3️⃣ evernotecid协议请求流程 (核心)
```
JavaScript: fetch('evernotecid://84D5E09F-B16A-446E-9E00-DB3B80E615C1/...')
        │
        ▼ (URL Rewrite Script)
JavaScript: fetch('/api/evernotecid/84D5E09F-B16A-446E-9E00-DB3B80E615C1/...')
        │
        ▼
┌─────────────────────────────────────────────────────────────────────┐
│                    Local HTTP Server                               │
│                                                                     │
│  ┌───────────────────────────────────────────────────────────────┐  │
│  │             API Proxy Handler                                 │  │
│  │                                                               │  │
│  │  Path: /api/evernotecid/*                                     │  │
│  │                                                               │  │
│  │  1. Extract evernotecid path                                  │  │
│  │  2. Create evernotecid:// URL                                 │  │
│  │  3. Use ENContentProtocol to fetch data                       │  │
│  │                                                               │  │
│  │  ┌─────────────────────────────────────────────────────────┐  │  │
│  │  │           ENContentProtocol                             │  │  │
│  │  │                                                         │  │  │
│  │  │  NSURLSession + Custom Protocol Class                  │  │  │
│  │  │                                                         │  │  │
│  │  │  URL: evernotecid://84D5E09F-B16A-446E-9E00-DB3B80E6..│  │  │◄─ ③
│  │  │                                                         │  │  │
│  │  │  ┌─────────────────────────────────────────────────┐   │  │  │
│  │  │  │  Account: self.accountController.account        │   │  │  │
│  │  │  │  UserID: [accountID intValue]                   │   │  │  │
│  │  │  │  ServiceHost: account.servicesProtocol.service..│   │  │  │
│  │  │  └─────────────────────────────────────────────────┘   │  │  │
│  │  │                                                         │  │  │
│  │  │  Returns: NSData + Content-Type                        │  │  │
│  │  └─────────────────────────────────────────────────────────┘  │  │
│  │                                                               │  │
│  │  4. Create HTTP Response with CORS headers                    │  │
│  │  5. Return data to WebView                                    │  │
│  └───────────────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────────────┘
        │
        ▼ Binary Data Response (images, documents, etc.)
     WebView JavaScript
```

## 🔄 完整请求响应时序图

```
WebView          Local HTTP Server       Remote Server       ENContentProtocol
   │                     │                     │                     │
   │ ①GET /index.html    │                     │                     │
   ├────────────────────►│                     │                     │
   │                     │ ②GET /index.html    │                     │
   │                     ├────────────────────►│                     │
   │                     │ ③HTML Response      │                     │
   │                     │◄────────────────────┤                     │
   │                     │ (inject script)     │                     │
   │ ④Modified HTML      │                     │                     │
   │◄────────────────────┤                     │                     │
   │                     │                     │                     │
   │ ⑤GET /superTable.js │                     │                     │
   ├────────────────────►│                     │                     │
   │                     │ ⑥GET /superTable.js │                     │
   │                     ├────────────────────►│                     │
   │                     │ ⑦JS File Response   │                     │
   │                     │◄────────────────────┤                     │
   │ ⑧JS File           │                     │                     │
   │◄────────────────────┤                     │                     │
   │                     │                     │                     │
   │ ⑨fetch(evernotecid://)                    │                     │
   │     ↓ (URL rewrite) │                     │                     │
   │ ⑩GET /api/evernotecid/*                   │                     │
   ├────────────────────►│                     │                     │
   │                     │                     │ ⑪evernotecid://    │
   │                     ├─────────────────────────────────────────►│
   │                     │                     │ ⑫Binary Data       │
   │                     │◄─────────────────────────────────────────┤
   │ ⑬HTTP Response      │                     │                     │
   │   (with CORS headers)│                    │                     │
   │◄────────────────────┤                     │                     │
```

## 🎯 关键组件职责

### ENWhiteboardHTTPServer
- **静态资源代理**：转发HTML/CSS/JS请求到远程服务器
- **URL重写注入**：自动在HTML中添加evernotecid重写逻辑  
- **API代理**：处理`/api/evernotecid/*`请求
- **CORS处理**：添加跨域头，统一origin

### 远程代理处理器 (Remote Proxy Handler)
- **请求转发**：将本地请求转发到远程服务器
- **内容修改**：对HTML内容注入URL重写脚本
- **错误处理**：处理网络超时、连接失败等情况
- **缓存控制**：设置适当的缓存策略

### URL重写脚本 (URL Rewrite Script)
```javascript
// 自动注入到所有HTML页面中的脚本
(function() {
    const isHTTPEnvironment = window.location.protocol === 'http:';
    
    if (isHTTPEnvironment) {
        // 重写fetch函数
        const originalFetch = window.fetch;
        window.fetch = function(resource, options) {
            if (typeof resource === 'string' && resource.startsWith('evernotecid://')) {
                const evernotecidPath = resource.substring('evernotecid://'.length);
                const newResource = `/api/evernotecid/${evernotecidPath}`;
                console.log('[WhiteboardJS] Converted fetch URL:', resource, '->', newResource);
                resource = newResource;
            }
            return originalFetch.call(this, resource, options);
        };
        
        // 重写XMLHttpRequest
        // ... (类似的重写逻辑)
    }
})();
```

## 💡 实现方案

### 1. 核心API设计

```objc
// ENWhiteboardHTTPServer.h
@interface ENWhiteboardHTTPServer : NSObject <GCDWebServerDelegate>

// 远程服务器URL配置
@property (nonatomic, strong, nullable) NSURL *remoteServerURL;

// 配置远程服务器
- (void)configureRemoteServer:(nullable NSURL *)remoteURL;

// 服务器状态信息
- (NSDictionary *)serverStatus;

@end
```

### 2. 使用方法

```objc
// 创建HTTP服务器
ENWhiteboardHTTPServer *httpServer = [[ENWhiteboardHTTPServer alloc] 
    initWithAccountController:accountController];

// 配置远程服务器
NSURL *remoteURL = [NSURL URLWithString:@"http://10.228.32.31:8888"];
[httpServer configureRemoteServer:remoteURL];

// 启动本地代理服务器
[httpServer startServer];

// 获取本地代理URL并加载
NSURL *localProxyURL = [httpServer indexHTMLURL];
[webView loadRequest:[NSURLRequest requestWithURL:localProxyURL]];
```

### 3. 工作模式

#### 本地文件模式 (默认)
```objc
// remoteServerURL = nil
[httpServer configureRemoteServer:nil];
// 服务本地common-database-mac目录的文件
```

#### 远程代理模式
```objc
// remoteServerURL = http://10.228.32.31:8888
NSURL *remoteURL = [NSURL URLWithString:@"http://10.228.32.31:8888"];
[httpServer configureRemoteServer:remoteURL];
// 代理远程服务器的所有内容
```

## 🔧 技术实现细节

### 1. 远程内容获取

```objc
- (GCDWebServerResponse *)handleRemoteProxyRequest:(GCDWebServerRequest *)request 
                                            forPath:(NSString *)path {
    // 构建远程URL
    NSURL *remoteURL = [NSURL URLWithString:path relativeToURL:self.remoteServerURL];
    
    // 异步请求远程资源
    NSURLRequest *remoteRequest = [NSURLRequest requestWithURL:remoteURL];
    NSURLSessionDataTask *task = [self.remoteSession dataTaskWithRequest:remoteRequest
                                                        completionHandler:^(NSData *data, NSURLResponse *response, NSError *error) {
        // 处理响应
    }];
    
    // 同步等待结果
    dispatch_semaphore_wait(semaphore, timeout);
}
```

### 2. HTML内容修改

```objc
- (NSString *)injectURLRewriteScript:(NSString *)html {
    NSString *rewriteScript = @"<script>/* URL重写逻辑 */</script>";
    
    // 在<head>标签后注入脚本
    NSRange headRange = [html rangeOfString:@"<head>" options:NSCaseInsensitiveSearch];
    if (headRange.location != NSNotFound) {
        NSUInteger insertLocation = headRange.location + headRange.length;
        NSMutableString *mutableHTML = [html mutableCopy];
        [mutableHTML insertString:rewriteScript atIndex:insertLocation];
        return [mutableHTML copy];
    }
    
    return html;
}
```

### 3. evernotecid协议处理

```objc
- (GCDWebServerResponse *)fetchEvernoteCidResource:(NSString *)evernoteCidURL {
    // 获取ENContentProtocol类
    EDAMUserID userID = [self.accountController.account.accountID intValue];
    Class protocolClass = [ENContentProtocol classForUserID:userID
                                               serviceHost:self.accountController.account.servicesProtocol.serviceHost];
    
    // 创建URLSession配置
    NSURLSessionConfiguration *config = [NSURLSessionConfiguration defaultSessionConfiguration];
    config.protocolClasses = @[protocolClass];
    
    // 执行请求
    NSURLSession *session = [NSURLSession sessionWithConfiguration:config];
    // ... 同步请求处理
}
```

## ✅ 优势总结

### 技术优势
1. **无需修改远程HTML**：自动注入URL重写逻辑
2. **透明代理**：WebView感知不到远程服务器的存在
3. **统一Origin**：所有资源都通过`localhost:54321`，解决CORS问题
4. **灵活配置**：支持本地文件和远程代理两种模式

### 部署优势
1. **简化部署**：HTML可以部署在任意HTTP服务器
2. **开发调试**：可以轻松切换本地和远程模式
3. **性能优化**：静态资源可以利用远程服务器的CDN等优化
4. **维护便利**：HTML更新无需重新打包应用

### 安全优势
1. **协议隔离**：evernotecid://协议始终在本地处理
2. **访问控制**：通过accountController控制资源访问权限
3. **CORS保护**：通过统一origin避免跨域安全问题

## 📊 性能考量

### 网络请求
- **延迟**：本地代理增加一层网络跳转，约增加1-2ms延迟
- **带宽**：远程资源通过本地代理，带宽使用量翻倍
- **缓存**：可配置适当的缓存策略减少重复请求

### 内存使用
- **HTML修改**：需要在内存中修改HTML内容，增加内存使用
- **并发连接**：支持多个并发代理请求，需要控制连接数

### 建议优化
1. **启用缓存**：对静态资源启用适当的缓存策略
2. **连接复用**：使用HTTP/1.1 keep-alive或HTTP/2
3. **压缩传输**：启用gzip压缩减少传输量
4. **异步处理**：避免阻塞主线程

---

*该文档描述了完整的远程HTML服务器支持架构，包含详细的原理图、数据流程和实现方案。通过该架构，Whiteboard可以完美支持部署在远程服务器的HTML内容，同时保持evernotecid://协议的正常工作。*