# Whiteboard WebView 跨域解决方案 - 集成指南

## 文件说明

本解决方案包含以下文件：

1. **Whiteboard_WebView_CrossOrigin_Solution.md** - 完整的方案文档
2. **ENWhiteboardEditorWKWebView_Implementation.h** - 头文件定义
3. **ENWhiteboardEditorWKWebView_Implementation.m** - 具体实现代码
4. **Whiteboard_Integration_Guide.md** - 本集成指南

## 快速集成步骤

### 1. 添加头文件引用

在 `ENWhiteboardEditorWKWebView.m` 文件顶部添加：

```objc
#import "ENWhiteboardEditorWKWebView_Implementation.h"
```

### 2. 修改初始化方法

在 `ENWhiteboardEditorWKWebView.m` 中修改初始化方法：

```objc
- (instancetype)initWithAccountController:(ENAccountController *)accountController {
    if (self = [super initWithAccountController:accountController]) {
        // 设置网络拦截器
        [self setupNetworkInterceptor];
    }
    return self;
}
```

### 3. 添加实现代码

将 `ENWhiteboardEditorWKWebView_Implementation.m` 中的代码添加到项目中，或者直接在 `ENWhiteboardEditorWKWebView.m` 中实现相关方法。

### 4. 测试验证

编译运行项目，查看控制台日志确认拦截器是否正常工作：

```
[Whiteboard Interceptor] Setting up network interceptor...
[Whiteboard Interceptor] Network interceptor setup completed
[Whiteboard Interceptor] Network interceptor initialized successfully
```

## 详细集成方案

### 方案A：扩展现有文件（推荐）

直接在现有的 `ENWhiteboardEditorWKWebView.m` 文件中添加实现：

```objc
// 在 ENWhiteboardEditorWKWebView.m 文件末尾添加

#pragma mark - Network Interceptor

- (void)setupNetworkInterceptor {
    // 将 Implementation.m 中的 setupNetworkInterceptor 方法代码复制到这里
}

- (NSString *)createNetworkInterceptorScript {
    // 将 Implementation.m 中的 createNetworkInterceptorScript 方法代码复制到这里
}

// ... 其他方法
```

### 方案B：独立实现文件

1. 将提供的 `.h` 和 `.m` 文件添加到 Xcode 项目中
2. 在 `ENWhiteboardEditorWKWebView.m` 中引用头文件
3. 确保在初始化时调用 `setupNetworkInterceptor`

## 调试和验证

### 1. 启用调试日志

```objc
// 在适当的地方调用
[self setDebugLoggingEnabled:YES];
```

### 2. 手动测试特定URL

```objc
// 测试特定的 evernotecid URL
[self validateEvernoteCidRequest:@"evernotecid://your-test-url"];
```

### 3. 查看拦截器状态

```objc
NSDictionary *status = [self getInterceptorStatus];
NSLog(@"Interceptor status: %@", status);
```

## 前端使用示例

在 `client.html` 或相关 JavaScript 文件中，可以正常使用 fetch：

```javascript
// 这样的代码现在应该能正常工作
async function loadEvernoteResource() {
    try {
        const response = await fetch('evernotecid://84D5E09F-B16A-446E-9E00-DB3B80E615C1/stage4yinxiangcom/10241/ENResource/p16');
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const buffer = await response.arrayBuffer();
        console.log('Resource loaded:', buffer.byteLength, 'bytes');
        
        // 可以进一步处理 buffer 数据
        return buffer;
        
    } catch (error) {
        console.error('Failed to load resource:', error);
        throw error;
    }
}

// 使用示例
loadEvernoteResource()
    .then(buffer => {
        // 处理获得的图片数据
        const blob = new Blob([buffer], { type: 'image/png' });
        const imageUrl = URL.createObjectURL(blob);
        
        const img = document.createElement('img');
        img.src = imageUrl;
        document.body.appendChild(img);
    })
    .catch(error => {
        console.error('Error:', error);
    });
```

## 常见问题和解决方案

### Q1: 拦截器没有生效怎么办？

**A:** 检查以下几点：
1. 确认 `setupNetworkInterceptor` 在 WebView 初始化时被调用
2. 查看控制台是否有相关的日志输出
3. 检查 `WKUserContentController` 是否正确设置了消息处理器

### Q2: 仍然出现跨域错误怎么办？

**A:** 可能的原因：
1. JavaScript 脚本注入时机不对，尝试使用 `WKUserScriptInjectionTimeAtDocumentStart`
2. WebKit 版本问题，可以尝试在 WebView 配置中设置额外的偏好项
3. 检查 `evernotecid://` 协议是否已正确注册

### Q3: 性能影响如何？

**A:** 
- JavaScript 拦截器只处理 `evernotecid://` 协议的请求
- 对其他请求没有额外开销
- 实际的数据传输仍通过原生的高效机制

### Q4: 如何在生产环境中禁用调试日志？

**A:** 在初始化后调用：
```objc
#if !DEBUG
[self setDebugLoggingEnabled:NO];
#endif
```

## 部署建议

### 开发阶段
1. 启用完整的调试日志
2. 使用测试URL验证功能
3. 在不同的 macOS 版本上测试

### 测试阶段
1. 测试实际的 whiteboard 应用场景
2. 验证各种类型的资源请求（图片、JSON、其他文件）
3. 性能测试，确保没有明显的性能下降

### 生产阶段
1. 禁用详细的调试日志
2. 保留错误日志用于问题诊断
3. 监控相关的崩溃和错误报告

## 后续优化建议

1. **缓存优化**：可以在 JavaScript 层面添加请求缓存机制
2. **批量处理**：对于大量小文件请求，可以考虑批量处理
3. **预加载**：对于已知的资源，可以实现预加载机制
4. **错误恢复**：增加更完善的错误恢复和降级机制

## 联系和支持

如果在集成过程中遇到问题，请：

1. 检查控制台日志
2. 使用提供的测试方法验证功能
3. 查看相关的通知和错误信息

这个解决方案应该能够有效解决 Whiteboard WebView 的跨域问题，让前端可以正常通过 fetch API 获取 `evernotecid://` 协议的资源数据。