# 快速HTML服务器部署指南

## 🎯 测试目标

为Whiteboard远程代理功能部署一个本地HTML测试服务器。

## 📁 创建测试目录

```bash
# 1. 创建测试目录
mkdir ~/whiteboard-test-server
cd ~/whiteboard-test-server

# 2. 创建测试文件
touch index.html client.html test.css superTable.js
```

## 📝 测试文件内容

### index.html
```html
<!DOCTYPE html>
<html>
<head>
    <title>Whiteboard远程测试服务器</title>
    <link rel="stylesheet" href="test.css">
</head>
<body>
    <h1>🌐 Whiteboard远程测试服务器</h1>
    <p>当前时间: <span id="time"></span></p>
    <p>这是来自远程服务器的HTML内容</p>
    
    <div class="test-section">
        <h2>🔧 功能测试</h2>
        <button onclick="testEvernotecid()">测试evernotecid://协议</button>
        <button onclick="testStaticResource()">测试静态资源</button>
        <div id="results"></div>
    </div>
    
    <script src="superTable.js"></script>
    <script>
        // 显示当前时间
        document.getElementById('time').textContent = new Date().toLocaleString();
        
        function testEvernotecid() {
            console.log('测试evernotecid://协议');
            const testURL = 'evernotecid://test-resource-id/sample/path';
            
            fetch(testURL)
                .then(response => {
                    document.getElementById('results').innerHTML = 
                        `<p style="color: green;">✅ evernotecid请求成功: ${response.status}</p>`;
                })
                .catch(error => {
                    document.getElementById('results').innerHTML = 
                        `<p style="color: red;">❌ evernotecid请求失败: ${error}</p>`;
                });
        }
        
        function testStaticResource() {
            console.log('测试静态资源加载');
            if (typeof remoteTestFunction === 'function') {
                document.getElementById('results').innerHTML += 
                    '<p style="color: green;">✅ JS文件加载成功</p>';
            } else {
                document.getElementById('results').innerHTML += 
                    '<p style="color: red;">❌ JS文件加载失败</p>';
            }
        }
    </script>
</body>
</html>
```

### client.html
```html
<!DOCTYPE html>
<html>
<head>
    <title>Whiteboard Client - Remote</title>
    <link rel="stylesheet" href="test.css">
</head>
<body>
    <h1>📋 Whiteboard Client (远程版本)</h1>
    <p>这是client.html的远程版本</p>
    
    <div class="feature-test">
        <h2>🧪 evernotecid协议测试</h2>
        <button onclick="loadResource()">加载evernotecid资源</button>
        <div id="resource-result"></div>
    </div>
    
    <script>
        console.log('远程client.html加载完成');
        
        function loadResource() {
            const resourceURL = 'evernotecid://84D5E09F-B16A-446E-9E00-DB3B80E615C1/resource/test';
            console.log('请求资源:', resourceURL);
            
            fetch(resourceURL)
                .then(response => response.arrayBuffer())
                .then(buffer => {
                    document.getElementById('resource-result').innerHTML = 
                        `<p>✅ 成功加载资源，大小: ${buffer.byteLength} 字节</p>`;
                })
                .catch(error => {
                    document.getElementById('resource-result').innerHTML = 
                        `<p>❌ 加载失败: ${error}</p>`;
                });
        }
    </script>
</body>
</html>
```

### test.css
```css
body {
    font-family: Arial, sans-serif;
    margin: 20px;
    background-color: #f5f5f5;
}

h1 {
    color: #2c3e50;
    border-bottom: 2px solid #3498db;
    padding-bottom: 10px;
}

.test-section, .feature-test {
    background: white;
    padding: 20px;
    margin: 20px 0;
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

button {
    background-color: #3498db;
    color: white;
    border: none;
    padding: 10px 20px;
    border-radius: 4px;
    cursor: pointer;
    margin: 5px;
}

button:hover {
    background-color: #2980b9;
}

#results, #resource-result {
    margin-top: 15px;
    padding: 10px;
    background-color: #ecf0f1;
    border-radius: 4px;
}
```

### superTable.js
```javascript
// 远程superTable.js测试文件
console.log('✅ 远程superTable.js加载成功');

// 测试函数
window.remoteTestFunction = function() {
    console.log('远程JS函数执行成功');
    return true;
};

// 模拟一些Whiteboard功能
window.WhiteboardRemote = {
    version: '1.0.0-remote',
    
    init: function() {
        console.log('远程Whiteboard初始化完成');
    },
    
    testEvernotecid: function(resourceId) {
        const url = `evernotecid://${resourceId}/test/data`;
        console.log('测试evernotecid URL:', url);
        
        return fetch(url)
            .then(response => response.arrayBuffer())
            .then(buffer => {
                console.log('evernotecid数据加载成功:', buffer.byteLength, '字节');
                return buffer;
            });
    }
};

// 自动初始化
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        WhiteboardRemote.init();
    });
} else {
    WhiteboardRemote.init();
}
```

## 🚀 启动服务器

### 方法1: Python (推荐)
```bash
cd ~/whiteboard-test-server
python3 -m http.server 8888 --bind 0.0.0.0
```

### 方法2: Node.js
```bash
cd ~/whiteboard-test-server
npx http-server -p 8888 -a 0.0.0.0 -c-1
```

### 方法3: PHP
```bash
cd ~/whiteboard-test-server
php -S 0.0.0.0:8888
```

## 🧪 测试配置

启动服务器后，你会看到类似输出：
```
Serving HTTP on 0.0.0.0 port 8888 (http://0.0.0.0:8888/) ...
```

### 获取你的IP地址
```bash
# macOS/Linux
ifconfig | grep "inet " | grep -v 127.0.0.1

# 或者
hostname -I
```

假设你的IP是 `192.168.1.100`，那么测试URL就是：
- http://192.168.1.100:8888/index.html
- http://192.168.1.100:8888/client.html

## 🔧 在Evernote中配置

```objc
// 在你的代码中使用获取到的IP
ENWhiteboardEditorWKWebView *webView = self.whiteboardWebView;
[webView loadRemoteWhiteboardFromURL:@"http://192.168.1.100:8888"];

// 或者使用localhost (如果在同一台机器上)
[webView loadRemoteWhiteboardFromURL:@"http://localhost:8888"];
```

## 📊 验证测试

### 1. 浏览器验证
直接在浏览器中访问：`http://你的IP:8888`

### 2. 检查服务器日志
服务器会显示所有请求：
```
192.168.1.100 - - [日期] "GET /index.html HTTP/1.1" 200 -
192.168.1.100 - - [日期] "GET /test.css HTTP/1.1" 200 -
192.168.1.100 - - [日期] "GET /superTable.js HTTP/1.1" 200 -
```

### 3. Evernote应用测试
在Evernote应用中执行：
```objc
[webView runComprehensiveRemoteTest];
```

查看控制台日志验证功能。

## 🔄 动态测试文件

为了更好地测试，你可以修改HTML文件，服务器会立即反映更改：

```bash
# 修改index.html添加时间戳
echo "<p>最后更新: $(date)</p>" >> ~/whiteboard-test-server/index.html
```

刷新浏览器或重新加载Evernote中的页面即可看到更改。

## 🚫 停止服务器

按 `Ctrl+C` 停止服务器。

## 💡 高级配置

### 使用不同端口
如果8888端口被占用：
```bash
python3 -m http.server 9999 --bind 0.0.0.0
```

### 添加HTTPS (可选)
```bash
# 生成自签名证书
openssl req -new -x509 -keyout server.pem -out server.pem -days 365 -nodes

# 使用HTTPS (需要自定义脚本)
```

这样你就有了一个完整的本地HTML测试服务器，可以全面测试Whiteboard的远程代理功能！