#!/bin/bash

# Whiteboard远程代理测试服务器一键部署脚本 (工程目录版本)

set -e

echo "🚀 开始部署Whiteboard测试服务器..."

# # 检测当前是否在工程目录
# if [[ ! -f "EverNote.xcworkspace" ]]; then
#     echo "❌ 请在Evernote工程根目录下运行此脚本"
#     echo "💡 当前目录: $(pwd)"
#     echo "📁 请切换到包含 EverNote.xcworkspace 的目录"
#     exit 1
# fi

echo "✅ 检测到Evernote工程目录: $(pwd)"

# 在工程目录下创建测试目录
TEST_DIR="$(pwd)/whiteboard-test-server"
echo "📁 创建测试目录: $TEST_DIR"
mkdir -p "$TEST_DIR"
cd "$TEST_DIR"

# 创建index.html
echo "📝 创建index.html..."
cat > index.html << 'EOF'
<!DOCTYPE html>
<html>
<head>
    <title>Whiteboard远程测试服务器</title>
    <link rel="stylesheet" href="test.css">
    <meta charset="UTF-8">
</head>
<body>
    <h1>🌐 Whiteboard远程测试服务器</h1>
    <div class="info-section">
        <p><strong>服务器时间:</strong> <span id="time"></span></p>
        <p><strong>页面URL:</strong> <span id="pageUrl"></span></p>
        <p><strong>状态:</strong> <span style="color: green;">✅ 远程服务器运行正常</span></p>
    </div>
    
    <div class="test-section">
        <h2>🔧 功能测试</h2>
        <button onclick="testEvernotecid()">测试evernotecid://协议</button>
        <button onclick="testStaticResource()">测试静态资源</button>
        <button onclick="runAllTests()">运行所有测试</button>
        <div id="results"></div>
    </div>
    
    <div class="test-section">
        <h2>📊 测试结果</h2>
        <div id="test-summary"></div>
    </div>
    
    <script src="superTable.js"></script>
    <script>
        // 更新页面信息
        function updatePageInfo() {
            document.getElementById('time').textContent = new Date().toLocaleString();
            document.getElementById('pageUrl').textContent = window.location.href;
        }
        
        updatePageInfo();
        setInterval(updatePageInfo, 1000);
        
        function log(message, type = 'info') {
            const results = document.getElementById('results');
            const timestamp = new Date().toLocaleTimeString();
            const color = type === 'success' ? 'green' : type === 'error' ? 'red' : 'blue';
            results.innerHTML += `<p style="color: ${color};">[${timestamp}] ${message}</p>`;
            console.log(`[RemoteTest] ${message}`);
        }
        
        function testEvernotecid() {
            log('开始测试evernotecid://协议...');
            const testURL = 'evernotecid://test-resource-id/sample/path';
            
            fetch(testURL)
                .then(response => {
                    if (response.ok) {
                        log(`✅ evernotecid请求成功: ${response.status}`, 'success');
                        return response.arrayBuffer();
                    } else {
                        throw new Error(`HTTP ${response.status}`);
                    }
                })
                .then(buffer => {
                    log(`📦 数据大小: ${buffer.byteLength} 字节`, 'success');
                })
                .catch(error => {
                    log(`❌ evernotecid请求失败: ${error.message}`, 'error');
                });
        }
        
        function testStaticResource() {
            log('测试静态资源加载...');
            
            if (typeof remoteTestFunction === 'function') {
                log('✅ superTable.js加载成功', 'success');
                remoteTestFunction();
            } else {
                log('❌ superTable.js加载失败', 'error');
            }
            
            // 测试CSS
            const computedStyle = window.getComputedStyle(document.body);
            if (computedStyle.fontFamily.includes('Arial')) {
                log('✅ test.css加载成功', 'success');
            } else {
                log('❌ test.css加载失败', 'error');
            }
        }
        
        function runAllTests() {
            log('========== 开始运行所有测试 ==========');
            document.getElementById('results').innerHTML = '';
            
            log('🌐 环境信息: ' + (window.location.protocol === 'http:' ? 'HTTP代理环境' : '文件协议环境'));
            
            setTimeout(() => testStaticResource(), 500);
            setTimeout(() => testEvernotecid(), 1500);
            setTimeout(() => {
                log('========== 所有测试完成 ==========');
                updateTestSummary();
            }, 3000);
        }
        
        function updateTestSummary() {
            const results = document.getElementById('results').innerHTML;
            const successCount = (results.match(/✅/g) || []).length;
            const errorCount = (results.match(/❌/g) || []).length;
            
            document.getElementById('test-summary').innerHTML = `
                <p><strong>测试统计:</strong></p>
                <p>✅ 成功: ${successCount}</p>
                <p>❌ 失败: ${errorCount}</p>
                <p>总计: ${successCount + errorCount}</p>
            `;
        }
        
        // 页面加载完成后自动检查
        window.addEventListener('load', () => {
            log('📱 远程测试页面加载完成');
            setTimeout(() => {
                if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
                    log('🏠 检测到本地访问');
                } else {
                    log('🌐 检测到远程访问');
                }
            }, 100);
        });
    </script>
</body>
</html>
EOF

# 创建client.html
echo "📝 创建client.html..."
cat > client.html << 'EOF'
<!DOCTYPE html>
<html>
<head>
    <title>Whiteboard Client - Remote</title>
    <link rel="stylesheet" href="test.css">
    <meta charset="UTF-8">
</head>
<body>
    <h1>📋 Whiteboard Client (远程版本)</h1>
    <div class="info-section">
        <p>这是从远程服务器加载的client.html</p>
        <p><strong>版本:</strong> Remote v1.0</p>
        <p><strong>加载时间:</strong> <span id="loadTime"></span></p>
    </div>
    
    <div class="feature-test">
        <h2>🧪 evernotecid协议测试</h2>
        <input type="text" id="resourceId" placeholder="输入资源ID" value="84D5E09F-B16A-446E-9E00-DB3B80E615C1">
        <button onclick="loadResource()">加载evernotecid资源</button>
        <button onclick="loadMultipleResources()">批量测试</button>
        <div id="resource-result"></div>
    </div>
    
    <div class="feature-test">
        <h2>📊 实时监控</h2>
        <p>请求计数: <span id="requestCount">0</span></p>
        <p>成功计数: <span id="successCount">0</span></p>
        <p>失败计数: <span id="errorCount">0</span></p>
    </div>
    
    <script>
        let requestCount = 0;
        let successCount = 0;
        let errorCount = 0;
        
        document.getElementById('loadTime').textContent = new Date().toLocaleString();
        
        function updateCounters() {
            document.getElementById('requestCount').textContent = requestCount;
            document.getElementById('successCount').textContent = successCount;
            document.getElementById('errorCount').textContent = errorCount;
        }
        
        function loadResource() {
            const resourceId = document.getElementById('resourceId').value || '84D5E09F-B16A-446E-9E00-DB3B80E615C1';
            const resourceURL = `evernotecid://${resourceId}/resource/test`;
            
            console.log('请求资源:', resourceURL);
            requestCount++;
            updateCounters();
            
            const startTime = Date.now();
            
            fetch(resourceURL)
                .then(response => {
                    const endTime = Date.now();
                    const duration = endTime - startTime;
                    
                    if (response.ok) {
                        successCount++;
                        return response.arrayBuffer().then(buffer => {
                            document.getElementById('resource-result').innerHTML = `
                                <div style="color: green;">
                                    <p>✅ 成功加载资源</p>
                                    <p>📦 数据大小: ${buffer.byteLength} 字节</p>
                                    <p>⏱️ 响应时间: ${duration}ms</p>
                                    <p>🔗 Content-Type: ${response.headers.get('content-type') || 'unknown'}</p>
                                </div>
                            `;
                        });
                    } else {
                        throw new Error(`HTTP ${response.status}`);
                    }
                })
                .catch(error => {
                    errorCount++;
                    document.getElementById('resource-result').innerHTML = `
                        <div style="color: red;">
                            <p>❌ 加载失败: ${error.message}</p>
                            <p>⏱️ 耗时: ${Date.now() - startTime}ms</p>
                        </div>
                    `;
                })
                .finally(() => {
                    updateCounters();
                });
        }
        
        function loadMultipleResources() {
            const testIds = [
                '84D5E09F-B16A-446E-9E00-DB3B80E615C1',
                'test-resource-1',
                'test-resource-2',
                'sample-id-123'
            ];
            
            document.getElementById('resource-result').innerHTML = '<p>🔄 批量测试进行中...</p>';
            
            let completed = 0;
            const results = [];
            
            testIds.forEach((id, index) => {
                setTimeout(() => {
                    const url = `evernotecid://${id}/test/batch-${index}`;
                    requestCount++;
                    updateCounters();
                    
                    fetch(url)
                        .then(response => {
                            successCount++;
                            results.push(`✅ ${id}: 成功 (${response.status})`);
                        })
                        .catch(error => {
                            errorCount++;
                            results.push(`❌ ${id}: 失败 (${error.message})`);
                        })
                        .finally(() => {
                            completed++;
                            updateCounters();
                            
                            if (completed === testIds.length) {
                                document.getElementById('resource-result').innerHTML = `
                                    <h3>批量测试结果:</h3>
                                    ${results.map(r => `<p>${r}</p>`).join('')}
                                `;
                            }
                        });
                }, index * 200);
            });
        }
        
        console.log('远程client.html加载完成');
    </script>
</body>
</html>
EOF

# 创建test.css
echo "🎨 创建test.css..."
cat > test.css << 'EOF'
body {
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
    margin: 0;
    padding: 20px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: #333;
    min-height: 100vh;
}

h1 {
    color: white;
    text-align: center;
    margin-bottom: 30px;
    text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
    font-size: 2.5em;
}

h2 {
    color: #2c3e50;
    border-bottom: 2px solid #3498db;
    padding-bottom: 10px;
    margin-top: 30px;
}

.info-section, .test-section, .feature-test {
    background: rgba(255, 255, 255, 0.95);
    padding: 25px;
    margin: 20px 0;
    border-radius: 12px;
    box-shadow: 0 8px 32px rgba(0,0,0,0.1);
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255,255,255,0.2);
}

button {
    background: linear-gradient(45deg, #3498db, #2980b9);
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 6px;
    cursor: pointer;
    margin: 8px 4px;
    font-size: 14px;
    font-weight: 500;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(52, 152, 219, 0.3);
}

button:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(52, 152, 219, 0.4);
    background: linear-gradient(45deg, #2980b9, #3498db);
}

button:active {
    transform: translateY(0);
}

input[type="text"] {
    padding: 10px;
    border: 2px solid #ddd;
    border-radius: 6px;
    margin: 5px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

input[type="text"]:focus {
    outline: none;
    border-color: #3498db;
    box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.1);
}

#results, #resource-result, #test-summary {
    margin-top: 20px;
    padding: 15px;
    background-color: #f8f9fa;
    border-radius: 8px;
    border-left: 4px solid #3498db;
    font-family: 'Monaco', 'Consolas', monospace;
    font-size: 13px;
    line-height: 1.6;
    max-height: 400px;
    overflow-y: auto;
}

#results p, #resource-result p {
    margin: 5px 0;
    padding: 5px 0;
}

.info-section p {
    margin: 10px 0;
    font-size: 16px;
}

.info-section strong {
    color: #2c3e50;
}

/* 滚动条样式 */
#results::-webkit-scrollbar, #resource-result::-webkit-scrollbar {
    width: 8px;
}

#results::-webkit-scrollbar-track, #resource-result::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 4px;
}

#results::-webkit-scrollbar-thumb, #resource-result::-webkit-scrollbar-thumb {
    background: #888;
    border-radius: 4px;
}

#results::-webkit-scrollbar-thumb:hover, #resource-result::-webkit-scrollbar-thumb:hover {
    background: #555;
}

/* 响应式设计 */
@media (max-width: 768px) {
    body {
        padding: 10px;
    }
    
    h1 {
        font-size: 2em;
    }
    
    .info-section, .test-section, .feature-test {
        padding: 15px;
        margin: 10px 0;
    }
    
    button {
        display: block;
        width: 100%;
        margin: 5px 0;
    }
    
    input[type="text"] {
        width: 100%;
        box-sizing: border-box;
    }
}
EOF

# 创建superTable.js
echo "⚡ 创建superTable.js..."
cat > superTable.js << 'EOF'
// Whiteboard远程superTable.js测试文件
// 版本: Remote Test v1.0

console.log('✅ 远程superTable.js加载成功');
console.log('📍 文件位置: 远程测试服务器');
console.log('🕐 加载时间:', new Date().toISOString());

// 全局测试函数
window.remoteTestFunction = function() {
    console.log('🧪 远程JS函数执行成功');
    return {
        status: 'success',
        message: '远程JavaScript功能正常',
        timestamp: new Date().toISOString()
    };
};

// 模拟Whiteboard功能
window.WhiteboardRemote = {
    version: '1.0.0-remote-test',
    startTime: Date.now(),
    
    init: function() {
        console.log('🚀 远程Whiteboard初始化开始');
        this.setupEventListeners();
        this.checkEnvironment();
        console.log('✅ 远程Whiteboard初始化完成');
    },
    
    setupEventListeners: function() {
        // 监听页面可见性变化
        document.addEventListener('visibilitychange', () => {
            console.log('👀 页面可见性变化:', document.visibilityState);
        });
        
        // 监听网络状态
        window.addEventListener('online', () => {
            console.log('🌐 网络连接已恢复');
        });
        
        window.addEventListener('offline', () => {
            console.log('📴 网络连接已断开');
        });
    },
    
    checkEnvironment: function() {
        const info = {
            userAgent: navigator.userAgent,
            platform: navigator.platform,
            language: navigator.language,
            cookieEnabled: navigator.cookieEnabled,
            onLine: navigator.onLine,
            protocol: window.location.protocol,
            host: window.location.host,
            pathname: window.location.pathname
        };
        
        console.log('🔍 环境信息:', info);
        return info;
    },
    
    testEvernotecid: function(resourceId) {
        const url = `evernotecid://${resourceId || 'test-resource-id'}/test/data`;
        console.log('🧪 测试evernotecid URL:', url);
        
        const startTime = performance.now();
        
        return fetch(url)
            .then(response => {
                const endTime = performance.now();
                const duration = Math.round(endTime - startTime);
                
                console.log(`⏱️ 请求耗时: ${duration}ms`);
                console.log('📊 响应状态:', response.status, response.statusText);
                console.log('📋 响应头:', Object.fromEntries(response.headers.entries()));
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
                return response.arrayBuffer();
            })
            .then(buffer => {
                const size = buffer.byteLength;
                console.log('📦 evernotecid数据加载成功:', size, '字节');
                
                // 分析数据类型
                const view = new Uint8Array(buffer.slice(0, 4));
                const header = Array.from(view).map(b => b.toString(16).padStart(2, '0')).join('');
                console.log('🔍 数据头部 (hex):', header);
                
                return {
                    buffer: buffer,
                    size: size,
                    header: header,
                    type: this.detectDataType(view)
                };
            })
            .catch(error => {
                console.error('❌ evernotecid请求失败:', error);
                throw error;
            });
    },
    
    detectDataType: function(header) {
        const headerHex = Array.from(header).map(b => b.toString(16).padStart(2, '0')).join('');
        
        if (headerHex.startsWith('ffd8ff')) return 'JPEG图片';
        if (headerHex.startsWith('89504e47')) return 'PNG图片';
        if (headerHex.startsWith('47494638')) return 'GIF图片';
        if (headerHex.startsWith('25504446')) return 'PDF文档';
        if (headerHex.startsWith('504b0304')) return 'ZIP压缩包';
        if (headerHex.startsWith('3c68746d')) return 'HTML文档';
        
        return '未知格式';
    },
    
    performanceTest: function(iterations = 10) {
        console.log(`🏃 开始性能测试，迭代次数: ${iterations}`);
        
        const results = [];
        const testResourceId = 'performance-test-' + Date.now();
        
        let completed = 0;
        
        for (let i = 0; i < iterations; i++) {
            const startTime = performance.now();
            
            this.testEvernotecid(testResourceId + '-' + i)
                .then(result => {
                    const endTime = performance.now();
                    const duration = endTime - startTime;
                    
                    results.push({
                        iteration: i + 1,
                        duration: duration,
                        size: result.size,
                        success: true
                    });
                })
                .catch(error => {
                    const endTime = performance.now();
                    const duration = endTime - startTime;
                    
                    results.push({
                        iteration: i + 1,
                        duration: duration,
                        success: false,
                        error: error.message
                    });
                })
                .finally(() => {
                    completed++;
                    
                    if (completed === iterations) {
                        this.analyzePerformanceResults(results);
                    }
                });
        }
    },
    
    analyzePerformanceResults: function(results) {
        const successful = results.filter(r => r.success);
        const failed = results.filter(r => !r.success);
        
        if (successful.length > 0) {
            const durations = successful.map(r => r.duration);
            const avgDuration = durations.reduce((a, b) => a + b, 0) / durations.length;
            const minDuration = Math.min(...durations);
            const maxDuration = Math.max(...durations);
            
            console.log('📊 性能测试结果:');
            console.log(`✅ 成功: ${successful.length}/${results.length}`);
            console.log(`⏱️ 平均耗时: ${avgDuration.toFixed(2)}ms`);
            console.log(`⚡ 最快: ${minDuration.toFixed(2)}ms`);
            console.log(`🐌 最慢: ${maxDuration.toFixed(2)}ms`);
        }
        
        if (failed.length > 0) {
            console.log(`❌ 失败: ${failed.length}/${results.length}`);
            failed.forEach(f => {
                console.log(`   第${f.iteration}次: ${f.error}`);
            });
        }
    },
    
    getStatus: function() {
        const uptime = Date.now() - this.startTime;
        return {
            version: this.version,
            uptime: uptime,
            ready: true,
            environment: this.checkEnvironment()
        };
    }
};

// 自动初始化
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        WhiteboardRemote.init();
    });
} else {
    WhiteboardRemote.init();
}

// 导出到全局作用域
window.WhiteboardRemote = WhiteboardRemote;

console.log('🎉 superTable.js远程版本加载完毕');
EOF

echo ""
echo "✅ 测试文件创建完成！"
echo "📁 测试目录: $TEST_DIR"
echo ""

# 检查Python是否可用
if command -v python3 &> /dev/null; then
    echo "🐍 检测到Python 3，推荐使用Python启动服务器"
    echo "💡 启动命令:"
    echo "   cd $TEST_DIR"
    echo "   python3 -m http.server 8888 --bind 0.0.0.0"
elif command -v python &> /dev/null; then
    echo "🐍 检测到Python 2，可以使用Python启动服务器"
    echo "💡 启动命令:"
    echo "   cd $TEST_DIR"
    echo "   python -m SimpleHTTPServer 8888"
fi

# 检查Node.js是否可用
if command -v node &> /dev/null; then
    echo "📦 检测到Node.js，可以使用http-server"
    echo "💡 启动命令:"
    echo "   cd $TEST_DIR"
    echo "   npx http-server -p 8888 -a 0.0.0.0 -c-1"
fi

# 检查PHP是否可用
if command -v php &> /dev/null; then
    echo "🐘 检测到PHP，可以使用PHP内置服务器"
    echo "💡 启动命令:"
    echo "   cd $TEST_DIR"
    echo "   php -S 0.0.0.0:8888"
fi

echo ""
echo "🏠 本地测试配置 (localhost):"
echo "📍 服务器地址: http://localhost:8888"
echo "📂 测试目录: $TEST_DIR"

echo ""
echo "🧪 使用方法:"
echo "1. 启动服务器:"
echo "   cd $TEST_DIR"
if command -v python3 &> /dev/null; then
    echo "   python3 -m http.server 8888"
    echo ""
    echo "🚀 快速启动 (推荐):"
    echo "   cd $TEST_DIR && python3 -m http.server 8888"
elif command -v python &> /dev/null; then
    echo "   python -m SimpleHTTPServer 8888"
    echo ""
    echo "🚀 快速启动:"
    echo "   cd $TEST_DIR && python -m SimpleHTTPServer 8888"
fi

echo ""
echo "2. 浏览器验证: http://localhost:8888"
echo "3. Evernote中配置:"
echo "   [webView loadRemoteWhiteboardFromURL:@\"http://localhost:8888\"]"
echo "4. 运行测试:"
echo "   [webView runComprehensiveRemoteTest]"

echo ""
echo "📝 添加到.gitignore (推荐):"
echo "   echo 'whiteboard-test-server/' >> .gitignore"

echo ""
echo "🎉 部署完成！测试服务器已在工程目录下创建"

# 创建启动脚本
echo "📝 创建启动脚本..."
cat > start-server.sh << 'SCRIPT_EOF'
#!/bin/bash

# Whiteboard测试服务器启动脚本

TEST_DIR="$(dirname "$0")"
cd "$TEST_DIR"

echo "🚀 启动Whiteboard测试服务器..."
echo "📂 当前目录: $(pwd)"
echo "📍 服务器地址: http://localhost:8888"
echo ""

# 检查端口是否被占用
if lsof -i :8888 >/dev/null 2>&1; then
    echo "⚠️  端口8888已被占用，尝试使用端口8889..."
    PORT=8889
else
    PORT=8888
fi

echo "🌐 访问地址: http://localhost:$PORT"
echo "🛑 按 Ctrl+C 停止服务器"
echo ""

# 启动服务器
if command -v python3 &> /dev/null; then
    echo "🐍 使用Python 3启动服务器..."
    python3 -m http.server $PORT
elif command -v python &> /dev/null; then
    echo "🐍 使用Python 2启动服务器..."
    python -m SimpleHTTPServer $PORT
elif command -v node &> /dev/null; then
    echo "📦 使用Node.js启动服务器..."
    npx http-server -p $PORT -a localhost -c-1
elif command -v php &> /dev/null; then
    echo "🐘 使用PHP启动服务器..."
    php -S localhost:$PORT
else
    echo "❌ 未找到可用的HTTP服务器"
    echo "请安装Python、Node.js或PHP"
    exit 1
fi
SCRIPT_EOF

chmod +x start-server.sh

echo ""
echo "✅ 已创建启动脚本: $TEST_DIR/start-server.sh"